#import "SPInfoContainerView.h"
#import "SPColorProvider.h"
#import "SPPreferencesController.h"
#import "SPInfoView.h"


@implementation SPInfoView


// ----------------------------------------------------------------------------
- (void) awakeFromNib
// ----------------------------------------------------------------------------
{
	[[SPPreferencesController sharedInstance] load];
	
	index = 0;
	collapsedHeight = 19.0f;
	height = 128.0f;
	container = (SPInfoContainerView*) [self superview];
	
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(containerBackgroundChanged:) name:SPInfoContainerBackgroundChangedNotification object:nil];
	[self containerBackgroundChanged:nil];
}


// ----------------------------------------------------------------------------
- (void) mouseDown:(NSEvent*)event
// ----------------------------------------------------------------------------
{
	NSPoint mousePosition = [event locationInWindow];
	NSPoint mousePositionInView = [self convertPoint:mousePosition fromView:nil];
	
	if ((mousePositionInView.y > ([self currentHeight] - 15.0f)) && ([event clickCount] > 1))
	{
		BOOL collapsed = !isCollapsed;
		[disclosureTriangle setState:collapsed ? 0 : 1];
		[self collapse:disclosureTriangle];
	}
}


// ----------------------------------------------------------------------------
- (void) drawRect:(NSRect)rect
// ----------------------------------------------------------------------------
{
	rect = [self bounds];

	NSColor* darkColor = nil;
	NSColor* brightColor = nil;
	if ([container hasDarkBackground])
	{
		darkColor = [NSColor colorWithDeviceRed:0.117f green:0.117f blue:0.117f alpha:1.0f];
		brightColor = [NSColor colorWithDeviceRed:0.321f green:0.321f blue:0.321f alpha:1.0f];
	}
	else
	{
		darkColor = [NSColor colorWithDeviceRed:0.381f green:0.381f blue:0.381f alpha:1.0f];
		brightColor = [NSColor colorWithDeviceRed:0.838f green:0.838f blue:0.838f alpha:1.0f];
	}

	float middle = rect.size.width * 0.5f;
	float ditchSize = 5.5f;
	float ypos = rect.origin.y + rect.size.height - 1.5f;

	[NSBezierPath setDefaultLineWidth:1.0f];
	NSBezierPath* path = [NSBezierPath bezierPath];
	
	[path moveToPoint:NSMakePoint(rect.origin.x, ypos)];
	[path lineToPoint:NSMakePoint(rect.origin.x + (middle - ditchSize), ypos)];
	[path lineToPoint:NSMakePoint(rect.origin.x + middle, ypos - ditchSize)];
	[path lineToPoint:NSMakePoint(rect.origin.x + (middle + ditchSize), ypos)];
	[path lineToPoint:NSMakePoint(rect.origin.x + rect.size.width, ypos)];
	
	[darkColor set];	
	[path stroke];

	ypos -= 1.0f;
	path = [NSBezierPath bezierPath];
	[path moveToPoint:NSMakePoint(rect.origin.x, ypos)];
	[path lineToPoint:NSMakePoint(rect.origin.x + (middle - ditchSize), ypos)];
	[path lineToPoint:NSMakePoint(rect.origin.x + middle, ypos - ditchSize)];
	[path lineToPoint:NSMakePoint(rect.origin.x + (middle + ditchSize), ypos)];
	[path lineToPoint:NSMakePoint(rect.origin.x + rect.size.width - 1.0f, ypos)];

	[brightColor set];	
	[path stroke];
	
	[super drawRect:rect];
}


// ----------------------------------------------------------------------------
- (IBAction) collapse:(id)sender
// ----------------------------------------------------------------------------
{
	int state = [sender state];

	// Better solution: disable disclosure cells of all info views during animation
	if ([container isAnimating])
	{
		[sender setState:(state == NSOnState) ? NSOffState : NSOnState];
		return;
	}
		
	if (state == NSOnState)
		[self setCollapsed:NO];
	else
		[self setCollapsed:YES];

	switch (index)
	{
		case TUNEINFO_CONTAINER_INDEX:
			gPreferences.mTuneInfoCollapsed = isCollapsed;
			break;
		case OSCILLOSCOPE_CONTAINER_INDEX:
			gPreferences.mOscilloscopeCollapsed = isCollapsed;
			break;
		case SIDREGISTER_CONTAINER_INDEX:
			gPreferences.mSidRegistersCollapsed = isCollapsed;
			break;
		case MIXER_CONTAINER_INDEX:
			gPreferences.mMixerCollapsed = isCollapsed;
			break;
		case FILTER_CONTAINER_INDEX:
			gPreferences.mFilterControlCollapsed = isCollapsed;
			break;
		case COMPOSER_CONTAINER_INDEX:
			gPreferences.mComposerPhotoCollapsed = isCollapsed;
			break;
	}

	[disclosureTriangle setEnabled:NO];
	[container positionSubviewsWithAnimation:YES];	
}


// ----------------------------------------------------------------------------
- (BOOL) isCollapsed
// ----------------------------------------------------------------------------
{
	return isCollapsed;
}


// ----------------------------------------------------------------------------
- (void) setCollapsed:(BOOL)flag
// ----------------------------------------------------------------------------
{
	isCollapsed = flag;
	[disclosureTriangle setState:flag ? 0 : 1];
}


// ----------------------------------------------------------------------------
- (float) currentHeight
// ----------------------------------------------------------------------------
{
	return isCollapsed ? collapsedHeight : height;
}


// ----------------------------------------------------------------------------
- (float) collapsedHeight
// ----------------------------------------------------------------------------
{
	return collapsedHeight;
}


// ----------------------------------------------------------------------------
- (float) height
// ----------------------------------------------------------------------------
{
	return height;
}


// ----------------------------------------------------------------------------
- (SPInfoContainerView*) container
// ----------------------------------------------------------------------------
{
	return container;
}


// ----------------------------------------------------------------------------
- (NSButton*) disclosureTriangle;
// ----------------------------------------------------------------------------
{
	return disclosureTriangle;
}


// ----------------------------------------------------------------------------
- (void) containerBackgroundChanged:(NSNotification *)aNotification
// ----------------------------------------------------------------------------
{
	if ([[disclosureTriangle cell] class] == [SPDisclosureCell class])
	{
		SPDisclosureCell* cell = [disclosureTriangle cell];

		if ([container hasDarkBackground])
		{
			[cell setBackgroundIsDark:YES];
			[cell setBackgroundColor:[container backgroundColor]];

			[titleText setTextColor:[NSColor whiteColor]];
		}
		else
		{
			[cell setBackgroundIsDark:NO];
			[titleText setTextColor:[NSColor blackColor]];
		}
	}
}


@end


#pragma mark -
@implementation SPDisclosureCell


static NSImage* SPHudDisclosureCollapsed = nil;
static NSImage* SPHudDisclosureExpanded = nil;
static NSImage* SPHudDisclosureTransient = nil;


// ----------------------------------------------------------------------------
- (void) awakeFromNib
// ----------------------------------------------------------------------------
{
	if (SPHudDisclosureCollapsed == nil)
	{
		SPHudDisclosureCollapsed = [NSImage imageNamed:@"disclosure_collapsed"];
		SPHudDisclosureExpanded = [NSImage imageNamed:@"disclosure_expanded"];
		SPHudDisclosureTransient = [NSImage imageNamed:@"disclosure_transient"];
	}
}


// ----------------------------------------------------------------------------
- (void) setBackgroundIsDark:(BOOL)flag
// ----------------------------------------------------------------------------
{
	backgroundIsDark = flag;
}


// ----------------------------------------------------------------------------
- (void) highlight:(BOOL)flag withFrame:(NSRect)cellFrame inView:(NSView *)controlView
// ----------------------------------------------------------------------------
{
	if (backgroundIsDark)
	{
		NSImage* image = [(NSButton*)controlView state] == 1 ? SPHudDisclosureExpanded : SPHudDisclosureCollapsed;
		NSRect imageRect = NSMakeRect(0.0f, 0.0f, [image size].width, [image size].height);
		NSRect imageFrame = imageRect;
		imageFrame.origin.x = (cellFrame.size.width - imageRect.size.width) / 2.0f;
		imageFrame.origin.y = (cellFrame.size.height - imageRect.size.height) / 2.0f;
			
		[image setFlipped:[controlView isFlipped]];
		if (flag)
		{
			[image drawInRect:imageFrame fromRect:imageRect operation:NSCompositePlusDarker fraction:1.0f];
			[image drawInRect:imageFrame fromRect:imageRect operation:NSCompositePlusDarker fraction:1.0f];
			[image drawInRect:imageFrame fromRect:imageRect operation:NSCompositePlusDarker fraction:1.0f];
		}
		else
			[image drawInRect:imageFrame fromRect:imageRect operation:NSCompositeSourceOver fraction:1.0f];
	}
	else
	{
		[super highlight:flag withFrame:cellFrame inView:controlView];
	}
}


// ----------------------------------------------------------------------------
- (void)drawWithFrame:(NSRect)cellFrame inView:(NSView *)controlView
// ----------------------------------------------------------------------------
{
	if (backgroundIsDark)
	{
		NSImage* image = [(NSButton*)controlView state] == 1 ? SPHudDisclosureExpanded : SPHudDisclosureCollapsed;
		NSRect imageRect = NSMakeRect(0.0f, 0.0f, [image size].width, [image size].height);
		NSRect imageFrame = imageRect;
		imageFrame.origin.x = (cellFrame.size.width - imageRect.size.width) / 2.0f;
		imageFrame.origin.y = (cellFrame.size.height - imageRect.size.height) / 2.0f;
			
		[image setFlipped:[controlView isFlipped]];
		[image drawInRect:imageFrame fromRect:imageRect operation:NSCompositeSourceOver fraction:1.0f];
	}
	else
		[super drawWithFrame:cellFrame inView:controlView];
}


@end

#pragma mark -
@implementation SPHudButtonCell

static NSImage* SPHudButtonLeftPart = nil;
static NSImage* SPHudButtonLeftSelectedPart = nil;
static NSImage* SPHudButtonMiddlePart = nil;
static NSImage* SPHudButtonMiddleSelectedPart = nil;
static NSImage* SPHudButtonRightPart = nil;
static NSImage* SPHudButtonRightSelectedPart = nil;


// ----------------------------------------------------------------------------
- (void)drawWithFrame:(NSRect)cellFrame inView:(NSView *)controlView
// ----------------------------------------------------------------------------
{
	SPInfoContainerView* container = [(SPInfoView*)[controlView superview] container];
	if ([container hasDarkBackground])
	{
		if (SPHudButtonLeftPart == nil)
		{
			SPHudButtonLeftPart = [NSImage imageNamed:@"HUD-ButtonLeft-N"];
			SPHudButtonLeftSelectedPart = [NSImage imageNamed:@"HUD-ButtonLeft-P"];
			SPHudButtonMiddlePart = [NSImage imageNamed:@"HUD-ButtonFill-N"];
			SPHudButtonMiddleSelectedPart = [NSImage imageNamed:@"HUD-ButtonFill-P"];
			SPHudButtonRightPart = [NSImage imageNamed:@"HUD-ButtonRight-N"];
			SPHudButtonRightSelectedPart = [NSImage imageNamed:@"HUD-ButtonRight-P"];
		}
		
		BOOL isSelected = [self isHighlighted];
		NSImage* leftImage = isSelected ? SPHudButtonLeftSelectedPart : SPHudButtonLeftPart;
		NSImage* middleImage = isSelected ? SPHudButtonMiddleSelectedPart : SPHudButtonMiddlePart;
		NSImage* rightImage = isSelected ? SPHudButtonRightSelectedPart : SPHudButtonRightPart;
		
		NSDrawThreePartImage(cellFrame, leftImage, middleImage, rightImage, NO, NSCompositeSourceOver, 1.0f, NO);
		
		NSMutableAttributedString *string = [[NSMutableAttributedString alloc] initWithAttributedString:[self attributedTitle]];
		[string removeAttribute:NSForegroundColorAttributeName range:NSMakeRange(0, [string length])];
		[string addAttribute:NSForegroundColorAttributeName value:[NSColor whiteColor] range:NSMakeRange(0, [string length])];
		[self setAttributedTitle:string];

		[super drawInteriorWithFrame:cellFrame inView:controlView];
	}
	else
	{
		NSMutableAttributedString *string = [[NSMutableAttributedString alloc] initWithAttributedString:[self attributedTitle]];
		[string removeAttribute:NSForegroundColorAttributeName range:NSMakeRange(0, [string length])];
		[string addAttribute:NSForegroundColorAttributeName value:[NSColor blackColor] range:NSMakeRange(0, [string length])];
		[self setAttributedTitle:string];

		[super drawWithFrame:cellFrame inView:controlView];
	}
}

@end


#pragma mark -
@implementation SPHudTableView


// ----------------------------------------------------------------------------
- (void) _sendDelegateWillDisplayCell:(id)cell forColumn:(id)column row:(int)row
// ----------------------------------------------------------------------------
{
    [super _sendDelegateWillDisplayCell:cell forColumn:column row:row];
    
    // Set text color
    if ([cell respondsToSelector:@selector(setTextColor:)])
	{
		if ([[SPColorProvider sharedInstance] providesDarkColors])
			[cell setTextColor:[NSColor whiteColor]];
		else
			[cell setTextColor:[NSColor blackColor]];
	}
}


// ----------------------------------------------------------------------------
- (id) _alternatingRowBackgroundColors
// ----------------------------------------------------------------------------
{
	return [[SPColorProvider sharedInstance] alternatingRowBackgroundColors];
}


// ----------------------------------------------------------------------------
- (id) _highlightColorForCell:(id)cell
// ----------------------------------------------------------------------------
{
	return [[SPColorProvider sharedInstance] highlightColor];
}

@end


#pragma mark -
@implementation SPHudSliderCell


static NSImage* SPHudSliderKnob = nil;
static NSImage* SPHudSliderKnobDisabled = nil;
static NSImage* SPHudSliderKnobPressed = nil;
static NSImage* SPHudSliderTrackLeft = nil;
static NSImage* SPHudSliderTrackMiddle = nil;
static NSImage* SPHudSliderTrackRight = nil;


// ----------------------------------------------------------------------------
- (void) awakeFromNib
// ----------------------------------------------------------------------------
{
	if (SPHudSliderKnob == nil)
	{
		SPHudSliderKnob = [NSImage imageNamed:@"hud_sliderknob"];
		SPHudSliderKnobDisabled = [NSImage imageNamed:@"hud_sliderknob_disabled"];
		SPHudSliderKnobPressed = [NSImage imageNamed:@"hud_sliderknob_pressed"];
		SPHudSliderTrackLeft = [NSImage imageNamed:@"hud_slider_track_left"];
		SPHudSliderTrackMiddle = [NSImage imageNamed:@"hud_slider_track_middle"];
		SPHudSliderTrackRight = [NSImage imageNamed:@"hud_slider_track_right"];
	}
}


// ----------------------------------------------------------------------------
- (void) drawKnob:(NSRect)rect
// ----------------------------------------------------------------------------
{
	SPInfoContainerView* container = [(SPInfoView*)[[self controlView] superview] container];
	if ([container hasDarkBackground])
	{
		NSImage* knobImage = [self isEnabled] ? ([self isHighlighted] ? SPHudSliderKnobPressed : SPHudSliderKnob) : SPHudSliderKnobDisabled;
		[knobImage compositeToPoint:NSMakePoint(rect.origin.x, rect.origin.y + rect.size.height + 1.0f) operation:NSCompositeSourceOver];
	}
	else
	{
		[super drawKnob:rect];
	}
}


// ----------------------------------------------------------------------------
- (void) drawBarInside:(NSRect)rect flipped:(BOOL)flipped
// ----------------------------------------------------------------------------
{
	SPInfoContainerView* container = [(SPInfoView*)[[self controlView] superview] container];
	if ([container hasDarkBackground])
	{
		[super drawBarInside:rect flipped:flipped];
		rect.origin.y += 3.0f;
		rect.size.height = 5.0f;
		NSDrawThreePartImage(rect, SPHudSliderTrackLeft, SPHudSliderTrackMiddle, SPHudSliderTrackRight, NO, NSCompositeSourceOver, 1.0f, flipped);
	}
	else
	{
		[super drawBarInside:rect flipped:flipped];
	}
}

@end


#pragma mark -
@implementation SPHudScroller

static NSImage* SPHudScrollerUp = nil;
static NSImage* SPHudScrollerUpSelected = nil;
static NSImage* SPHudScrollerDown = nil;
static NSImage* SPHudScrollerDownSelected = nil;
static NSImage* SPHudScrollerBackTop = nil;
static NSImage* SPHudScrollerBack = nil;
static NSImage* SPHudScrollerBackBottom = nil;
static NSImage* SPHudScrollerKnobTop = nil;
static NSImage* SPHudScrollerKnobMiddle = nil;
static NSImage* SPHudScrollerKnobBottom = nil;
static NSImage* SPHudScrollerFill = nil;

enum
{
    NSScrollerPartWhole = 0, 
    NSScrollerPartUpperExtra = 1, 
    NSScrollerPartKnob = 2, 
    NSScrollerPartLowerExtra = 3, 
    NSScrollerPartUpArrow = 4, 
    NSScrollerPartDownArrow = 5, 
    NSScrollerPartWithoutArrow = 6
};


// ----------------------------------------------------------------------------
- (void) awakeFromNib
// ----------------------------------------------------------------------------
{
	if (SPHudScrollerUp == nil)
	{
		SPHudScrollerUp = [NSImage imageNamed:@"hud_scroller_up"];
		SPHudScrollerUpSelected = [NSImage imageNamed:@"hud_scroller_up_selected"];
		SPHudScrollerDown = [NSImage imageNamed:@"hud_scroller_down"];
		SPHudScrollerDownSelected = [NSImage imageNamed:@"hud_scroller_down_selected"];
		SPHudScrollerBackTop = [NSImage imageNamed:@"hud_scroller_back_top"];
		SPHudScrollerBack = [NSImage imageNamed:@"hud_scroller_back"];
		SPHudScrollerBackBottom = [NSImage imageNamed:@"hud_scroller_back_bottom"];
		SPHudScrollerKnobTop = [NSImage imageNamed:@"hud_scroller_knob_top"];
		SPHudScrollerKnobMiddle = [NSImage imageNamed:@"hud_scroller_knob_middle"];
		SPHudScrollerKnobBottom = [NSImage imageNamed:@"hud_scroller_knob_bottom"];
		SPHudScrollerFill = [NSImage imageNamed:@"hud_scroller_fill"];
	}
}


// ----------------------------------------------------------------------------
- (void) drawArrow:(NSScrollerArrow)arrow highlightPart:(int)part
// ----------------------------------------------------------------------------
{
    NSRect bounds = [self bounds];
    BOOL flipped = [self isFlipped];
    
	NSRect imageRect;
	NSRect incrementArrowRect = [self rectForPart:NSScrollerIncrementLine];
	NSRect decrementArrowRect = [self rectForPart:NSScrollerDecrementLine];
	NSRect slotRect = [self rectForPart:NSScrollerKnobSlot];
	
	// Slot rect tells us how much room there is above and belove the slot, so we can use that
	// to find out where double arrow buttons are, so we draw them correctly in every mode
	BOOL doubleOnTop = slotRect.origin.y > 25.0f;
	BOOL doubleOnBottom = ([self bounds].size.height - (slotRect.origin.y + slotRect.size.height)) > 25.0f;

	BOOL incrementSelected = part == NSScrollerIncrementArrow;
	BOOL decrementSelected = part == NSScrollerDecrementArrow;
	
	//NSLog(@"arrow: %d, part: %d, inc: %@, dec: %@, slot: %@, top:%d, bottom:%d\n", arrow, part, 
	//		NSStringFromRect(incrementArrowRect), NSStringFromRect(decrementArrowRect), NSStringFromRect(slotRect),
	//		doubleOnTop, doubleOnBottom);
	
	flipped = !flipped;

	// Lower increment button
	NSImage* image = incrementSelected ? SPHudScrollerUpSelected : SPHudScrollerUp;
	imageRect.origin = NSZeroPoint;
	imageRect.size = [image size];

	if ([image isFlipped] != flipped)
		[image setFlipped:flipped];

	[image drawInRect:incrementArrowRect fromRect:imageRect operation:NSCompositeCopy fraction:1.0f];
	if (doubleOnBottom)
	{
		// Double decrement button
		NSRect doubleArrowRect;
		doubleArrowRect = incrementArrowRect;
		doubleArrowRect.size.height = 14.0f;
		doubleArrowRect.origin.y -= 14.0f;
		image = decrementSelected ? SPHudScrollerDownSelected : SPHudScrollerDown;
		imageRect.origin = NSZeroPoint;
		imageRect.size = [image size];

		if ([image isFlipped] != flipped)
			[image setFlipped:flipped];

		[image drawInRect:doubleArrowRect fromRect:imageRect operation:NSCompositeCopy fraction:1.0f];
	}
	
	flipped = !flipped;

	// Upper decrement button
	image = decrementSelected ? SPHudScrollerUpSelected : SPHudScrollerUp;
	imageRect.origin = NSZeroPoint;
	imageRect.size = [image size];

	if ([image isFlipped] != flipped)
		[image setFlipped:flipped];

	[image drawInRect:decrementArrowRect fromRect:imageRect operation:NSCompositeCopy fraction:1.0f];
	if (doubleOnTop)
	{
		// Double increment button
		NSRect doubleArrowRect;
		doubleArrowRect = decrementArrowRect;
		doubleArrowRect.size.height = 14.0f;
		doubleArrowRect.origin.y += 14.0f;
		image = incrementSelected ? SPHudScrollerDownSelected : SPHudScrollerDown;
		imageRect.origin = NSZeroPoint;
		imageRect.size = [image size];

		if ([image isFlipped] != flipped)
			[image setFlipped:flipped];

		[image drawInRect:doubleArrowRect fromRect:imageRect operation:NSCompositeCopy fraction:1.0f];
	}
}


// ----------------------------------------------------------------------------
- (void) drawKnob
// ----------------------------------------------------------------------------
{
    BOOL flipped = [self isFlipped];
    NSRect rect = [self rectForPart:NSScrollerPartKnob]; 
	rect = NSInsetRect(rect, 1.0f, 4.0f);
	NSDrawThreePartImage(rect, SPHudScrollerKnobTop, SPHudScrollerKnobMiddle, SPHudScrollerKnobBottom, YES, NSCompositeSourceOver, 1.0f, flipped);
}


// ----------------------------------------------------------------------------
- (void) drawKnobSlotInRect:(NSRect)rect highlight:(BOOL)highlight
// ----------------------------------------------------------------------------
{
    NSRect imageRect;
    imageRect.origin = NSZeroPoint;
    imageRect.size = [SPHudScrollerFill size];
	NSRect fillRect = rect;
	fillRect.origin.y -= 6.0f;
	fillRect.size.height += 10.0f;
    [SPHudScrollerFill drawInRect:fillRect fromRect:imageRect operation:NSCompositeCopy fraction:1.0f];

    BOOL flipped = [self isFlipped];
	rect = NSInsetRect(rect, 0.0f, 2.0f);
	NSDrawThreePartImage(rect, SPHudScrollerBackTop, SPHudScrollerBack, SPHudScrollerBackBottom, YES, NSCompositeSourceOver, 1.0f, flipped);
}


@end




#pragma mark -
@implementation SPHudPopUpButtonCell

static NSImage* SPHudPopupLeft = nil;
static NSImage* SPHudPopupMiddle = nil;
static NSImage* SPHudPopupRight = nil;
static NSImage* SPHudPopupSelectedLeft = nil;
static NSImage* SPHudPopupSelectedMiddle = nil;
static NSImage* SPHudPopupSelectedRight = nil;


// ----------------------------------------------------------------------------
- (void) awakeFromNib
// ----------------------------------------------------------------------------
{
	if (SPHudPopupLeft == nil)
	{
		SPHudPopupLeft = [NSImage imageNamed:@"hud_popup_left"];
		SPHudPopupMiddle = [NSImage imageNamed:@"hud_popup_middle"];
		SPHudPopupRight = [NSImage imageNamed:@"hud_popup_right"];
		SPHudPopupSelectedLeft = [NSImage imageNamed:@"hud_popup_selected_left"];
		SPHudPopupSelectedMiddle = [NSImage imageNamed:@"hud_popup_selected_middle"];
		SPHudPopupSelectedRight = [NSImage imageNamed:@"hud_popup_selected_right"];
	}
}


// ----------------------------------------------------------------------------
- (void) drawBezelWithFrame:(NSRect)frame inView:(NSView*)controlView
// ----------------------------------------------------------------------------
{
    BOOL flipped = [controlView isFlipped];
	frame.size.height = 17.0f;
	NSDrawThreePartImage(frame, SPHudPopupLeft, SPHudPopupMiddle, SPHudPopupRight, NO, NSCompositeSourceOver, 1.0f, flipped);
}


// ----------------------------------------------------------------------------
- (NSRect) drawTitle:(NSAttributedString*)title withFrame:(NSRect)frame inView:(NSView*)controlView
// ----------------------------------------------------------------------------
{
	NSMutableAttributedString *string = [[NSMutableAttributedString alloc] initWithAttributedString:[self attributedTitle]];
	[string removeAttribute:NSForegroundColorAttributeName range:NSMakeRange(0, [string length])];
	[string addAttribute:NSForegroundColorAttributeName value:[NSColor whiteColor] range:NSMakeRange(0, [string length])];

	return [super drawTitle:string withFrame:frame inView:controlView];
}


@end
