/*
HMTabViewItem.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMAppKitEx.h"
#import "HMTabItemGroupView.h"
#import "HMTabItemView.h"
#import "HMTabView.h"
#import "HMTabViewItem.h"
#import "HMWindow.h"

static NSDictionary*    _labelAttr = nil;
static NSDictionary*    _selectedLabelAttr = nil;
static NSDictionary*    _noThumbnailLabelAttr = nil;

@implementation HMTabViewItem

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (void)load
{
    NSAutoreleasePool*  pool;
    pool = [[NSAutoreleasePool alloc] init];
    
    // Get resources
    if (!_labelAttr) {
        NSMutableParagraphStyle* paragraphStyle;
        paragraphStyle = [[NSMutableParagraphStyle alloc] init];
        [paragraphStyle setLineBreakMode:NSLineBreakByTruncatingTail];
        [paragraphStyle setAlignment:NSCenterTextAlignment];
        
        NSShadow*   shadow;
        shadow = [[NSShadow alloc] init];
        [shadow setShadowOffset:NSMakeSize(0.0, -1.0f)];
        [shadow setShadowBlurRadius:0.0f];
        [shadow setShadowColor:[[NSColor blackColor] colorWithAlphaComponent:0.5f]];
        
        NSShadow*   noThumbnailShadow;
        noThumbnailShadow = [[NSShadow alloc] init];
        [noThumbnailShadow setShadowOffset:NSMakeSize(0.0, -1.0f)];
        [noThumbnailShadow setShadowBlurRadius:0.0f];
        [noThumbnailShadow setShadowColor:[NSColor colorWithCalibratedWhite:0.90f alpha:1.0f]];
        
        _labelAttr = [[NSDictionary dictionaryWithObjectsAndKeys:
                [NSFont boldSystemFontOfSize:11.0], NSFontAttributeName, 
                [NSColor colorWithCalibratedWhite:0.12f alpha:1.0f], NSForegroundColorAttributeName, 
                paragraphStyle, NSParagraphStyleAttributeName, 
                nil] retain];
        _selectedLabelAttr = [[NSDictionary dictionaryWithObjectsAndKeys:
                [NSFont boldSystemFontOfSize:11.0], NSFontAttributeName, 
                [NSColor whiteColor], NSForegroundColorAttributeName, 
                shadow, NSShadowAttributeName, 
                paragraphStyle, NSParagraphStyleAttributeName, 
                nil] retain];
        _noThumbnailLabelAttr = [[NSDictionary dictionaryWithObjectsAndKeys:
                [NSFont boldSystemFontOfSize:11.0], NSFontAttributeName, 
                [NSColor colorWithCalibratedWhite:0.25f alpha:1.0f], NSForegroundColorAttributeName, 
                noThumbnailShadow, NSShadowAttributeName, 
                paragraphStyle, NSParagraphStyleAttributeName, 
                nil] retain];
        
        [paragraphStyle release];
        [shadow release];
    }
    
    [pool release];
}

- (id)initWithIdentifier:(id)identifier
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize member variables
    _identifier = [identifier retain];
    _showTitle = NO;
    _isWorking = NO;
    _estimatedProgress = 0.0f;
    _hasCloseButton = NO;
    
    return self;
}

- (void)dealloc
{
    [_identifier release], _identifier = nil;
    [_label release], _label = nil;
    [_attributedLabel release], _attributedLabel = nil;
    [_selectedAttributedLabel release], _selectedAttributedLabel = nil;
    [_noThumbnailAttributedLabel release], _noThumbnailAttributedLabel = nil;
    [_view release], _view = nil;
    
    [_thumbnailImage release];
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Label --
//--------------------------------------------------------------//

- (NSString*)label
{
    return [[_label copy] autorelease];
}

- (NSAttributedString*)attributedLabel
{
    return _attributedLabel;
}

- (NSAttributedString*)selectedAttributedLabel
{
    return _selectedAttributedLabel;
}

- (NSAttributedString*)noThumbnailAttributedLabel
{
    return _noThumbnailAttributedLabel;
}

- (void)setLabel:(NSString*)label
{
    [_label release];
    _label = [label copy];
    
    // Create attributed label
    [_attributedLabel release];
    [_selectedAttributedLabel release];
    [_noThumbnailAttributedLabel release];
    
    _attributedLabel = [[NSAttributedString alloc] 
            initWithString:_label attributes:_labelAttr];
    _selectedAttributedLabel = [[NSAttributedString alloc] 
            initWithString:_label attributes:_selectedLabelAttr];
    _noThumbnailAttributedLabel = [[NSAttributedString alloc] 
            initWithString:_label attributes:_noThumbnailLabelAttr];
}

- (NSString*)subLabel
{
    return [[_subLabel copy] autorelease];
}

- (void)setSubLabel:(NSString*)subLabel
{
    [_subLabel release];
    _subLabel = [subLabel copy];
}

- (NSImage*)thumbnailImage
{
    return _thumbnailImage;
}

- (void)setThumbnailImage:(NSImage*)image
{
    [_thumbnailImage release];
    _thumbnailImage = [image retain];
    
    // Check flip
    BOOL    isFlipped;
    isFlipped = [_tabItemView isFlipped];
    if ([_thumbnailImage isFlipped] != isFlipped) {
        [_thumbnailImage setFlipped:isFlipped];
    }
    
    // Update appearance
    [_tabItemView setNeedsDisplay:YES];
}

- (BOOL)isWorking
{
    return _isWorking;
}

- (void)setWorking:(BOOL)working
{
    if (_isWorking == working) {
        return;
    }
    
    _isWorking = working;
    
    // Start progress indicator animation
    if ([_tabView hasThumbnail]) {
        NSProgressIndicator*    nsIndicator;
        nsIndicator = [_tabItemView nsProgressIndicator];
        
        if (working) {
            [nsIndicator startAnimation:self];
            [nsIndicator setDoubleValue:0.0f];
        }
        else {
            [nsIndicator stopAnimation:self];
        }
    }
    else {
        HMProgressIndicator*    hmIndicator;
        hmIndicator = [_tabItemView hmProgressIndicator];
        
        if (working) {
            [hmIndicator startAnimation:self];
            [hmIndicator setDoubleValue:0.0f];
        }
        else {
            [hmIndicator stopAnimation:self];
        }
    }
    
    // Update appearance
    [_tabItemView setNeedsDisplay:YES];
}

- (float)estimatedProgress
{
    return _estimatedProgress;
}

- (void)setEstimatedProgress:(float)estimatedProgress
{
    _estimatedProgress = estimatedProgress;
    
    // Set progress
    if ([_tabView hasThumbnail]) {
        NSProgressIndicator*    nsIndicator;
        nsIndicator = [_tabItemView nsProgressIndicator];
        if (_estimatedProgress > 0.0f) {
            [nsIndicator stopAnimation:self];
        }
    }
    else {
        HMProgressIndicator*    hmIndicator;
        hmIndicator = [_tabItemView hmProgressIndicator];
        [hmIndicator setDoubleValue:_estimatedProgress];
        
        if ([hmIndicator isHidden]) {
            [hmIndicator setHidden:NO];
        }
    }
    
    // Update appearance
    [_tabItemView setNeedsDisplay:YES];
}

//--------------------------------------------------------------//
#pragma mark -- Identifier --
//--------------------------------------------------------------//

- (id)identifier
{
    return _identifier;
}

- (void)setIdentifier:(id)identifier
{
    [_identifier release];
    _identifier = [identifier retain];
}

//--------------------------------------------------------------//
#pragma mark -- View --
//--------------------------------------------------------------//

- (NSView*)view
{
    if (_nsTabViewItem) {
        return [_nsTabViewItem view];
    }
    
    return _view;
}

- (void)setView:(NSView*)view
{
    [_view release];
    _view = [view retain];
}

//--------------------------------------------------------------//
#pragma mark -- Status --
//--------------------------------------------------------------//

- (HMTabViewItemStatus)status
{
    return _status;
}

- (void)setStatus:(HMTabViewItemStatus)status
{
    _status = status;
}

//--------------------------------------------------------------//
#pragma mark -- Parent tab view --
//--------------------------------------------------------------//

- (HMTabView*)tabView
{
    return _tabView;
}

- (void)setTabView:(HMTabView*)tabView
{
    _tabView = tabView;
}

- (HMTabItemView*)tabItemView
{
    return _tabItemView;
}

- (void)setTabItemView:(HMTabItemView*)tabItemView
{
    _tabItemView = tabItemView;
}

- (NSTabViewItem*)nsTabViewItem
{
    return _nsTabViewItem;
}

- (void)setNSTabViewItem:(NSTabViewItem*)nsTabViewItem
{
    _nsTabViewItem = nsTabViewItem;
}

//--------------------------------------------------------------//
#pragma mark -- Close button --
//--------------------------------------------------------------//

- (BOOL)hasCloseButton
{
    return _hasCloseButton;
}

- (void)setHasCloseButton:(BOOL)hasCloseButton
{
    _hasCloseButton = hasCloseButton;
}

@end
