/**********************************************************************
Copyright (c) 2003 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.List;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Expression for binary object.
 *
 * @version $Revision: 1.6 $
 **/
public class BinaryExpression extends ScalarExpression
{
    /**
     * Constructor
     * @param qs the QueryExpression
     */
    protected BinaryExpression(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * 
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     */
    public BinaryExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        super(qs, mapping, te);
    }

    /**
     * Generates statement as e.g. FUNCTION_NAME(arg[,argN])
     * @param functionName
     * @param args ScalarExpression list
     */
    public BinaryExpression(String functionName, List args)
    {
        super(functionName, args);
    }

    /**
     * Performs a function on two binary expressions.
     * op(operand1,operand2)
     * operand1 op operand2 
     * @param operand1 the first expression
     * @param op the operator between operands
     * @param operand2 the second expression
     */
    public BinaryExpression(ScalarExpression operand1, DyadicOperator op, ScalarExpression operand2)
    {
        super(operand1, op, operand2);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.eq(this);
        }
        else if (expr instanceof BinaryExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.noteq(this);
        }
        else if (expr instanceof BinaryExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof BinaryExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);
        }
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof BinaryExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);
        }
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof BinaryExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);
        }
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof BinaryExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);
        }
        else
        {
            return super.gteq(expr);
        }
    }

    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof BinaryExpression)
        {
            return new BinaryExpression(this, OP_CONCAT, expr);
        }
        else
        {
            return super.add(expr);
        }
    }

    public BooleanExpression in(ScalarExpression expr)
    {
        return new BooleanExpression(this, OP_IN, expr);
    }
}
