/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.mapped.mapping.OIDMapping;
import org.datanucleus.util.ClassUtils;

/**
 * Representation of an expression "new MyObject(param1, param2)" as the result in a JDOQL query.
 * @version $Revision: 1.7 $
 */
public class NewObjectExpression extends ScalarExpression
{
    /** The class that we need to construct an instance of. */
    Class newClass = null;

    /** The arguments to use in the constructor. */
    List ctrArgs = null;
    
    final Constructor ctr; 

    /**
     * Constructor.
     * @param qs The query expression
     * @param cls The class to construct an instance of
     * @param args the constructor args
     */
    public NewObjectExpression(QueryExpression qs, Class cls, List args)
    {
        super(qs);
        this.newClass = cls;
        ctrArgs = new ArrayList(args);

        // Check that this constructor exists before going any further
        Class[] ctrTypes = new Class[args.size()];
        for (int i=0;i<args.size();i++)
        {
            if( ((ScalarExpression)args.get(i)).getMapping() instanceof OIDMapping )
            {
                ctrTypes[i] = qs.getClassLoaderResolver().classForName(((OIDMapping)((ScalarExpression)args.get(i)).getMapping()).getType());
            }
            else if( args.get(i) instanceof NewObjectExpression )
            {
                ctrTypes[i] = ((NewObjectExpression)args.get(i)).getNewClass();
            }
            else
            {
                ctrTypes[i] = ((ScalarExpression)args.get(i)).getMapping().getJavaType();
            }
        }
        ctr = ClassUtils.getConstructorWithArguments(newClass, ctrTypes);
        if (ctr == null)
        {
            throw new NucleusUserException(LOCALISER.msg("037013", newClass.getName()+typeList(Arrays.asList(ctrTypes))));
        }
    }

    private static String typeList(List exprs)
    {
        StringBuffer s = new StringBuffer("(");
        Iterator i = exprs.iterator();

        while (i.hasNext())
        {
            s.append(i.next()).append(i.hasNext() ? ',' : ')');
        }

        return s.toString();
    }
    
    /**
     * Accessor for the class of which we should create a new instance.
     * @return The class
     */
    public Class getNewClass()
    {
        return newClass;
    }

    /**
     * Accessor for the constructor argument expressions.
     * @return List of constructor argument expressions
     */
    public List getArgumentExpressions()
    {
        return ctrArgs;
    }

    /**
     * Method to return the new object using the passed values for the arguments.
     * @param values The values of the arguments
     * @return The new object
     */
    public Object createNewObject(Object[] values)
    {
        if ((values == null || values.length == 0) && ctrArgs.size() > 0)
        {
            throw new NucleusUserException(LOCALISER.msg("037014", "0", "" + ctrArgs.size()));
        }
        if (values.length != ctrArgs.size())
        {
            throw new NucleusUserException(LOCALISER.msg("037014", "" + values.length, "" + ctrArgs.size()));
        }

        Object obj = null;
        if (ctr != null)
        {
            try
            {
                obj = ctr.newInstance(values);
            }
            catch (Exception e)
            {
                throw new NucleusUserException(LOCALISER.msg("037015", newClass.getName(), e));
            }
        }
        return obj;
    }
}