/**********************************************************************
Copyright (c) 2012 Google Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 **********************************************************************/
package com.google.appengine.datanucleus;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * A set of utility methods for serializing BigDecimals into sortable strings.
 *
 *  This class encodes and decodes BigDecimals into strings that preserve the
 * sort order of the BigDecimal. In order to preserve the sort order when
 * strings are compared lexographically they need to be a fixed length. For this
 * reason this class imposes a limit on the precision used. This is set
 * arbitrarily to 100 decimal digits.
 *
 *  The format of the stored string is as follows:
 *
 *  [-_][-_+]XXXXXXXX,XXXXXXXX...
 *
 *  The first character is either '-' for negative numbers or '_' for positive
 * numbers. Followed by a sign byte and 8 hex characters that represent the
 * exponent of the number. The sign byte for the exponent is not needed for
 * decoding, as the hex value contains the sign information. It purely exists to
 * preserve sorting. Because a negative number with a large exponent is smaller
 * than one with a small exponent, for negative numbers the exponent is negated.
 * (Meaning that --00000001 represents an exponent of -1 for a negative number)
 *
 *  Following this there is a ',' and then the un-scaled value of the BigDecimal
 * written in hex as though it were a fixed size two's complement number with
 * 336 bits precision. (This is derived from the number of hex digits required
 * to represent 100 decimal digits of precision.) So for a positive number this
 * will simply be the number zero padded for 84 hex digits. For negative numbers
 * it is first subtracted from the constant 2^336 and then written in hex.
 *
 */
public final class BigDecimals {

  public static final int MAX_PRECISION = 100;

  private static final int RADIX = 16;

  private static final int STORED_VALUE_SIZE = 84; // Log16(10^100)

  static final int STRING_SIZE = STORED_VALUE_SIZE + 8 + 3;

  private static final BigInteger MAX_VALUE = maxValue();

  private static final BigDecimal ZERO = new BigDecimal(0);

  private BigDecimals() {
  }

  private static BigInteger maxValue() {
    StringBuffer sb = new StringBuffer(STORED_VALUE_SIZE + 1);
    sb.append('1');
    for (int i = 0; i < STORED_VALUE_SIZE; i++) {
      sb.append('0');
    }
    return new BigInteger(sb.toString(), RADIX);
  }

  /**
   * @param sortFormatedString
   *          A string generated by {@link #toSortableString(BigDecimal)}
   *
   * @return A BigDecimal that is numerically equal to the original BigDecimal
   *         used to create the provided string. (Note that there may be a
   *         reduction in precision if there were trailing zeros in the original
   *         number)
   */
  public static BigDecimal fromSortableString(String sortFormattedString) {
    if (sortFormattedString.length() != STRING_SIZE) {
      throw new IllegalArgumentException("Expected string of length: "
          + STRING_SIZE + " but received one of "
          + sortFormattedString.length());
    }
    char sign = sortFormattedString.charAt(0);
    long scale = Long.parseLong(sortFormattedString.substring(2, 10), RADIX);
    BigInteger value;
    if ('-' == sign) {
      value = MAX_VALUE.subtract(
          new BigInteger(sortFormattedString.substring(11), RADIX));
      value = value.negate();
    } else if ('_' == sign) {
      value = new BigInteger(sortFormattedString.substring(11), RADIX);
      scale = -scale;
    } else {
      throw new IllegalArgumentException(
          "Corrupt BigDecimal string: " + sortFormattedString);
    }
    BigDecimal result = new BigDecimal(value, (int) scale);
    return result.stripTrailingZeros();
  }

  /**
   * Transforms a BigDecimal into a 95 character string such that for any two
   * given BigDecimals if a > b then the string generated by a will be
   * lexicographically before b and vice versa. Note that this property means
   * that if a is numerically equal to b, they will encode to the same string.
   * Thus there is no preservation of the precision to which the number was
   * stored, only the value.
   *
   * @param number
   *          The value to encode into a string.
   * @return A sort-preserving string representation that may be converted back
   *         into a BigDecimal by using the {@link #fromSortableString(String)}
   *         method.
   * @throws IllegalArgumentException
   *           If the value passed in has more than 100 digits of precision, or
   *           if there are leading or trailing zeros that cannot be removed by
   *           scaling the number, because the scale is already so close to
   *           Integer.MAX_VALUE or Integer.MIN_VALUE that scaling it to remove
   *           the zeros would cause the integer to wrap.
   */
  public static String toSortableString(BigDecimal number) {
    StringBuffer sb = new StringBuffer(STRING_SIZE);
    checkScale(number);
    BigDecimal num = canonicalize(number);
    int scale = num.scale();
    String value;
    if (num.signum() == -1) {
      sb.append('-');
      if (scale > 0) {
        sb.append('-');
      } else {
        sb.append('+');
      }
      value = MAX_VALUE.subtract(num.unscaledValue().abs()).toString(RADIX);
    } else {
      sb.append('_');
      if (scale > 0) {
        sb.append('-');
      } else {
        sb.append('_');
      }
      scale = -scale;
      value = num.unscaledValue().abs().toString(RADIX);
    }
    sb.append(getScaleString(scale));
    sb.append(',');
    for (int i = 0; i < STORED_VALUE_SIZE - value.length(); i++) {
      sb.append('0');
    }
    sb.append(value);
    assert sb.length() == STRING_SIZE;
    return sb.toString();
  }

  private static void checkScale(BigDecimal number) {
    if (number.precision() > MAX_PRECISION) {
      throw new IllegalArgumentException("Requested to store number with: "
          + number.precision()
          + " decimals of precision which is more than the max of: "
          + MAX_PRECISION);
    }
  }

  private static String getScaleString(int scale) {
    return String.format("%08x", scale);
  }

  private static BigDecimal canonicalize(BigDecimal number) {
    if (number.compareTo(ZERO) == 0) {
      return ZERO;
    }
    BigDecimal result = number.stripTrailingZeros();
    int addedPrecision = MAX_PRECISION - result.precision();
    if (addedPrecision > 0) {
      long newScale = ((long) result.scale() + addedPrecision);
      if (newScale > Integer.MAX_VALUE) {
        throw new IllegalArgumentException(
            "Attempted to serialize an number with a scale of: "
            + number.scale()
            + " this is too close to Integer.MAX_VALUE to connicalized.");
      }
      return result.setScale((int) newScale);
    } else {
      return result;
    }
  }

}
