/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2002 Mike Martin (TJDO)
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.math.BigInteger;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a Byte literal.
 */
public class ByteLiteral extends NumericExpression implements Literal
{
    private final BigInteger value;

    /** Raw value that this literal represents. */
    Object rawValue;

    /**
     * Creates a byte literal
     * @param qs the QueryExpression
     * @param mapping the mapping
     * @param value the literal value
     */    
    public ByteLiteral(QueryExpression qs, JavaTypeMapping mapping, BigInteger value)
    {
        super(qs);
        this.mapping = mapping;

        this.value = value;

        mapping = qs.getStoreManager().getMappingManager().getMappingWithDatastoreMapping(
            Byte.class, false, false, qs.getClassLoaderResolver());        
        st.appendParameter(mapping, value);
        
    }

    public Object getValue()
    {
        return value;
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((ByteLiteral)expr).value) == 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((ByteLiteral)expr).value) != 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((ByteLiteral)expr).value) < 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);
        }            
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((ByteLiteral)expr).value) <= 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);
        }            
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((ByteLiteral)expr).value) > 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);
        }            
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((ByteLiteral)expr).value) >= 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);
        }            
        else
        {
            return super.gteq(expr);
        }
    }

    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new ByteLiteral(qs, mapping, value.add(((ByteLiteral)expr).value));
        }
        else
        {
            return super.add(expr);
        }
    }

    public ScalarExpression sub(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new ByteLiteral(qs, mapping, value.subtract(((ByteLiteral)expr).value));
        }
        else
        {
            return super.sub(expr);
        }
    }

    public ScalarExpression mul(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new ByteLiteral(qs, mapping, value.multiply(((ByteLiteral)expr).value));
        }
        else
        {
            return super.mul(expr);
        }
    }

    public ScalarExpression div(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new ByteLiteral(qs, mapping, value.divide(((ByteLiteral)expr).value));
        }
        else
        {
            return super.div(expr);
        }
    }

    public ScalarExpression mod(ScalarExpression expr)
    {
        if (expr instanceof ByteLiteral)
        {
            return new ByteLiteral(qs, mapping, value.mod(((ByteLiteral)expr).value));
        }
        else
        {
            return super.mod(expr);
        }
    }

    public ScalarExpression neg()
    {
        return new ByteLiteral(qs, mapping, value.negate());
    }

    /**
     * Method to save a "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @param val The raw value
     */
    public void setRawValue(Object val)
    {
        this.rawValue = val;
    }

    /**
     * Accessor for the "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @return The raw value
     */
    public Object getRawValue()
    {
        return rawValue;
    }
}