/**********************************************************************
Copyright (c) 2002 Kelly Grizzle and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2002 Mike Martin - unknown changes
2003 Andy Jefferson - commented
    ...
**********************************************************************/
package org.datanucleus.jpa.state;

import org.datanucleus.FetchPlan;
import org.datanucleus.StateManager;
import org.datanucleus.Transaction;
import org.datanucleus.state.LifeCycleState;

/**
 * Class representing the life cycle state of PersistentClean.
 */
class PersistentClean extends LifeCycleState
{
    /** Protected Constructor to prevent external instantiation. */
    protected PersistentClean()
    {
		isPersistent = true;        	
        isDirty = false;
        isNew = false;
        isDeleted = false;
        isTransactional = true;
        
        stateType = P_CLEAN;
    }

    /**
     * Method to transition to delete persistent.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionDeletePersistent(StateManager sm)
    {
        return changeState(sm, P_DELETED);
    }

    /**
     * Method to transition to nontransactional.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionMakeNontransactional(StateManager sm)
    {
        sm.clearSavedFields();
        return changeState(sm, P_NONTRANS);
    }

    /**
     * Method to transition to transient.
     * @param sm StateManager.
     * @param useFetchPlan to make transient the fields in the fetch plan
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionMakeTransient(StateManager sm, boolean useFetchPlan, boolean detachAllOnCommit)
    {
        if (useFetchPlan)
        {
            sm.loadUnloadedFieldsInFetchPlan();
        }
        return changeState(sm, TRANSIENT);
    }

    /**
     * Method to transition to commit state.
     * @param sm StateManager.
     * @param tx the Transaction been committed.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionCommit(StateManager sm, Transaction tx)
    {
        sm.clearSavedFields();

        if (tx.getRetainValues())
        {
            return changeState(sm, P_NONTRANS);
        }
        else
        {
            sm.clearNonPrimaryKeyFields();
            return changeState(sm, HOLLOW);
        }
    }

    /**
     * Method to transition to rollback state.
     * @param sm StateManager.
     * @param tx The Transaction
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRollback(StateManager sm, Transaction tx)
    {
        if (tx.getRestoreValues())
        {
            sm.restoreFields();
            return changeState(sm, P_NONTRANS);
        }
        else
        {
            sm.clearNonPrimaryKeyFields();
            sm.clearSavedFields();
            return changeState(sm, HOLLOW);
        }
    }

    /**
     * Method to transition to evict state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionEvict(StateManager sm)
    {
        sm.clearNonPrimaryKeyFields();
        sm.clearSavedFields();
        return changeState(sm, HOLLOW);
    }

    /**
     * Method to transition to write-field state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionWriteField(StateManager sm)
    {
        Transaction tx = sm.getObjectManager().getTransaction();
        if (tx.getRestoreValues())
        {
            sm.saveFields();
        }

        return changeState(sm, P_DIRTY);
    }

	/**
	 * Method to transition to refresh state.
	 * @param sm StateManager.
	 * @return new LifeCycle state.
	 **/
	public LifeCycleState transitionRefresh(StateManager sm)
	{
		sm.clearSavedFields();

        // Refresh the FetchPlan fields and unload all others
        sm.refreshFieldsInFetchPlan();
        sm.unloadNonFetchPlanFields();

        Transaction tx = sm.getObjectManager().getTransaction();
		if (tx.isActive())
		{
			return changeState(sm,P_CLEAN);
		}
		return changeState(sm,P_NONTRANS);      
	}
	
    /**
     * Method to transition to retrieve state.
     * @param sm StateManager.
	 * @param fgOnly only the current fetch group fields
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRetrieve(StateManager sm, boolean fgOnly)
    {
		if (fgOnly)
        {
            sm.loadUnloadedFieldsInFetchPlan();
        }
		else
        {
			sm.loadUnloadedFields();
        }
        return this;
    }

    /**
     * Method to transition to retrieve state.
     * @param sm StateManager.
     * @param fetchPlan the fetch plan to load fields
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRetrieve(StateManager sm, FetchPlan fetchPlan)
    {
        sm.loadUnloadedFieldsOfClassInFetchPlan(fetchPlan);
        return this;
    }
    
    /**
     * Method to transition to detached-clean.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionDetach(StateManager sm)
    {
        return changeState(sm, DETACHED_CLEAN);
    }

    /**
     * Method to return a string version of this object.
     * @return The string "P_CLEAN".
     **/
    public String toString()
    {
        return "P_CLEAN";
    }
}