// Copyright 2014 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.syntax;

import java.io.IOException;

/**
 * Base class for all expression nodes in the AST.
 */
public abstract class Expression extends ASTNode {

  /**
   * Kind of the expression. This is similar to using instanceof, except that it's more efficient
   * and can be used in a switch/case.
   */
  public enum Kind {
    BINARY_OPERATOR,
    COMPREHENSION,
    CONDITIONAL,
    DICTIONARY_LITERAL,
    DOT,
    FUNCALL,
    IDENTIFIER,
    INDEX,
    INTEGER_LITERAL,
    LIST_LITERAL,
    SLICE,
    STRING_LITERAL,
    UNARY_OPERATOR,
  }

  /**
   * Returns the result of evaluating this build-language expression in the
   * specified environment. All BUILD language datatypes are mapped onto the
   * corresponding Java types as follows:
   *
   * <pre>
   *    int   -> Integer
   *    float -> Double          (currently not generated by the grammar)
   *    str   -> String
   *    [...] -> List&lt;Object>    (mutable)
   *    (...) -> List&lt;Object>    (immutable)
   *    {...} -> Map&lt;Object, Object>
   *    func  -> Function
   * </pre>
   *
   * @return the result of evaluting the expression: a Java object corresponding
   *         to a datatype in the BUILD language.
   * @throws EvalException if the expression could not be evaluated.
   * @throws InterruptedException may be thrown in a sub class.
   */
  public final Object eval(Environment env) throws EvalException, InterruptedException {
    try {
      if (Callstack.enabled) {
        Callstack.push(this);
      }
      try {
        return doEval(env);
      } catch (EvalException ex) {
        throw maybeTransformException(ex);
      }
    } finally {
      if (Callstack.enabled) {
        Callstack.pop();
      }
    }
  }

  /**
   * Evaluates the expression and returns the result.
   *
   * <p>This method is only invoked by the super class {@link Expression} when calling {@link
   * #eval(Environment)}.
   *
   * @throws EvalException if the expression could not be evaluated
   * @throws InterruptedException may be thrown in a sub class.
   */
  abstract Object doEval(Environment env) throws EvalException, InterruptedException;

  @Override
  public final void prettyPrint(Appendable buffer, int indentLevel) throws IOException {
    prettyPrint(buffer);
  }

  /**
   * Expressions should implement this method instead of {@link #prettyPrint(Appendable, int)},
   * since the {@code indentLevel} argument is not needed.
   */
  @Override
  public abstract void prettyPrint(Appendable buffer) throws IOException;

  /**
   * Kind of the expression. This is similar to using instanceof, except that it's more efficient
   * and can be used in a switch/case.
   */
  public abstract Kind kind();
}
