---
layout: documentation
title: Query how-to
---

<a name="_Bazel_Query_How_To"> </a>
# Bazel query how-to

This is a quick tutorial to get you started using Bazel's query language to
trace dependencies in your code.

For a language details and `--output` flag details, please see the reference
manual, [Bazel query reference](query.html).  You can get help for Bazel query
by typing `bazel help query`.

To execute a query while ignoring errors such as missing targets, use the
`--keep_going` flag.

<a name="_Contents"></a>
## Contents

*   [Finding the Dependencies of a Rule](#Finding_the_Dependencies_of_a_Ru)
*   [Tracing the Dependency Chain between Two
    Packages](#Tracing_the_Dependency_Chain_bet)
    *   [Aside: implicit dependencies](#Aside_implicit_dependencies)
*   [Reverse Dependencies](#Reverse_Dependencies)
*   [Miscellaneous Uses](#Miscellaneous_Uses)
    *   [What exists ...](#What_exists_)
        *   [What packages exist beneath
            `foo`?](#What_packages_exist_beneath_foo_)
        *   [What rules are defined in the `foo`
            package?](#What_rules_are_defined_in_the_foo)
        *   [What files are generated by rules in the `foo`
            package?](#What_files_are_generated_by_rule)
        *   [What's the set of BUILD files needed to build
            `//foo`?](#What_s_the_set_of_BUILD_files_ne)
        *   [What are the individual tests that a `test_suite` expands
            to?](#What_are_the_individual_tests_th)
            *   [Which of those are C++ tests?](#Which_of_those_are_C_tests_)
            *   [Which of those are small? Medium?
                Large?](#Which_of_those_are_small_Medium_)
        *   [What are the tests beneath `foo` that match a
            pattern?](#What_are_the_tests_beneath_foo_t)
        *   [What package contains file
            `src/main/java/com/example/cache/LRUCache.java`?
            ](#What_package_contains_file_java_)
        *   [What is the build label for
            `src/main/java/com/example/cache/LRUCache.java`?
            ](#What_is_the_build_label_for_java)
        *   [What build rule contains file
            `src/main/java/com/example/cache/LRUCache.java` as a
            source?](#What_build_rule_contains_file_ja)
    *   [What package dependencies exist ...](#What_package_dependencies_exist_)
        *   [What packages does `foo` depend on? (What do I need to check out
            to build `foo`)](#What_packages_does_foo_depend_on)
        *   [What packages does the `foo` tree depend on, excluding
            `foo/contrib`?](#What_packages_does_the_foo_)
    *   [What rule dependencies exist ...](#What_rule_dependencies_exist_)
        *   [What genproto rules does bar depend
            upon?](#What_genproto_rules_does_bar_)
        *   [Find the definition of some JNI (C++) library that is transitively
            depended upon by a Java binary rule in the servlet
            tree.](#Find_the_definition_of_some_JNI_)
            *   [...Now find the definitions of all the Java binaries that
                depend on them](#_Now_find_the_definitions_of_all)
    *   [What file dependencies exist ...](#What_file_dependencies_exist_)
        *   [What's the complete set of Java source files required to build
            QUX?](#What_s_the_complete_set_of_Java_)
        *   [What is the complete set of Java source files required to build
            QUX's tests?](#What_is_the_complete_set_of_Java)
    *   [What differences in dependencies between X and Y exist
        ...](#What_differences_in_dependencies)
        *   [What targets does `//foo` depend on that `//foo:foolib` does
            not?](#What_targets_does_foo_depend_on_)
        *   [What C++ libraries do the `foo` tests depend on that the `//foo`
            production binary does _not_ depend
            on?](#What_C_libraries_do_the_foo_test)
    *   [Why does this dependency exist ...](#Why_does_this_dependency_exist_)
        *   [Why does `bar` depend on
            `groups2`?](#Why_does_bar_depend_on_groups)
        *   [Show me a path from `docker/updater:updater_systest` (a `py_test`)
            to some `cc_library` that it depends
            upon:](#Show_me_a_path_from_docker_updater)
        *   [Why does library `//photos/frontend:lib` depend on two variants of
            the same library `//third_party/jpeglib` and
            `//third_party/jpeg`?](#Why_does_library_photos_fronten)
    *   [What depends on ...](#What_depends_on_)
        *   [What rules under bar depend on Y?](#What_rules_under_bar_depend_o)
    *   [How do I break a dependency ...](#How_do_I_break_a_dependency_)
        *   [What dependency paths do I have to break to make `bar` no longer
            depend on X?](#What_dependency_paths_do_I_have_)
    *   [Misc ...](#Misc_)
        *   [How many sequential steps are there in the `ServletSmokeTests`
            build?](#How_many_sequential_steps_are_th)

<a name="Finding_the_Dependencies_of_a_Ru"></a>
## Finding the Dependencies of a Rule

To see the dependencies of `//src/main/java/com/example/base:base`, use the
`deps` function in bazel query:

```
$ bazel query "deps(src/main/java/com/example/base:base)"
//resources:translation.xml
//src/main/java/com/example/base:AbstractPublishedUri.java
...
```

This is the set of all targets required to build <code>//src/main/java/com/example/base:base</code>.

<a name="Tracing_the_Dependency_Chain_bet"></a>
## Tracing the Dependency Chain between Two Packages

The library `//third_party/zlib:zlibonly` isn't in the BUILD file for
`//src/main/java/com/example/base`, but it is an indirect dependency. How can
we trace this dependency path?  There are two useful functions here:
`allpaths` and `somepath`

```
$ bazel query "somepath(src/main/java/com/example/base:base, third_party/zlib:zlibonly)"
//src/main/java/com/example/base:base
//translations/tools:translator
//translations/base:base
//third_party/py/MySQL:MySQL
//third_party/py/MySQL:_MySQL.so
//third_party/mysql:mysql
//third_party/zlib:zlibonly
$ bazel query "allpaths(src/main/java/com/example/common/base:base, third_party/...)"
  ...many errors detected in BUILD files...
//src/main/java/com/example/common/base:base
//third_party/java/jsr166x:jsr166x
//third_party/java/sun_servlet:sun_servlet
//src/main/java/com/example/common/flags:flags
//src/main/java/com/example/common/flags:base
//translations/tools:translator
//translations/tools:aggregator
//translations/base:base
//tools/pkg:pex
//tools/pkg:pex_phase_one
//tools/pkg:pex_lib
//third_party/python:python_lib
//translations/tools:messages
//third_party/py/xml:xml
//third_party/py/xml:utils/boolean.so
//third_party/py/xml:parsers/sgmlop.so
//third_party/py/xml:parsers/pyexpat.so
//third_party/py/MySQL:MySQL
//third_party/py/MySQL:_MySQL.so
//third_party/mysql:mysql
//third_party/openssl:openssl
//third_party/zlib:zlibonly
//third_party/zlib:zlibonly_v1_2_3
//third_party/python:headers
//third_party/openssl:crypto
```

<a name="Aside_implicit_dependencies"></a>
### Aside: implicit dependencies

The BUILD file for `src/main/java/com/example/common/base` never references
`//translations/tools:aggregator`. So, where's the direct dependency?

Certain rules include implicit dependencies on additional libraries or tools.
For example, to build a `genproto` rule, you need first to build the Protocol
Compiler, so every `genproto` rule carries an implicit dependency on the
protocol compiler.  These dependencies are not mentioned in the build file,
but added in by the build tool.  The full set of implicit dependencies is
currently undocumented; read the source code of
[`RuleClassProvider`](https://github.com/bazelbuild/bazel/tree/master/src/main/java/com/google/devtools/build/lib/packages/RuleClassProvider.java).

<a name="Reverse_Dependencies"></a>
## Reverse Dependencies

You might want to know the set of targets that depends on some target.  e.g.,
if you're going to change some code, you might want to know what other code
you're about to break.  You can use `rdeps(u, x)` to find the reverse
dependencies of the targets in `x` within the transitive closure of `u`.

Unfortunately, invoking, e.g., `rdeps(..., daffie/annotations2:constants-lib)`
is not practical for a large tree, because it requires parsing every BUILD file
and building a very large dependency graph (Bazel may run out of memory).  If
you would like to execute this query across a large repository, you may have to
query subtrees and then combine the results.

<a name="Miscellaneous_Uses"></a>
## Miscellaneous Uses

You can use `bazel query` to analyze many dependency relationships.

<a name="What_exists_"></a>
### What exists ...

<a name="What_packages_exist_beneath_foo_"></a>
#### What packages exist beneath `foo`?

```sh
bazel query 'foo/...' --output package
```

<a name="What_rules_are_defined_in_the_gw"></a>
#### What rules are defined in the `foo` package?

```sh
bazel query 'kind(rule, foo:all)' --output label_kind
```

<a name="What_files_are_generated_by_rule"></a>
#### What files are generated by rules in the `foo` package?

```sh
bazel query 'kind("generated file", //foo:*)'
```

<a name="What_s_the_set_of_BUILD_files_ne"></a>
#### What's the set of BUILD files needed to build `//foo`?

bazel query 'buildfiles(deps(//foo))' --output location | cut -f1 -d:

<a name="What_are_the_individual_tests_th"></a>
#### What are the individual tests that a `test_suite` expands to?

```sh
bazel query 'tests(//foo:smoke_tests)'
```

<a name="Which_of_those_are_C_tests_"></a>
##### Which of those are C++ tests?

```sh
bazel query 'kind(cc_.*, tests(//foo:smoke_tests))'
```

<a name="Which_of_those_are_small_Medium_"></a>
#### Which of those are small?  Medium?  Large?

```sh
bazel query 'attr(size, small, tests(//foo:smoke_tests))'

bazel query 'attr(size, medium, tests(//foo:smoke_tests))'

bazel query 'attr(size, large, tests(//foo:smoke_tests))'
```

<a name="What_are_the_tests_beneath_foo_t"></a>
#### What are the tests beneath `foo` that match a pattern?

```sh
bazel query 'filter("pa?t", kind(".*_test rule", //foo/...))'
```

The pattern is a regex and is applied to the full name of the rule. It's similar to doing

```sh
bazel query 'kind(".*_test rule", //foo/...)' | grep -E 'pa?t'
```

<a name="What_package_contains_file_java_"></a>
#### What package contains file src/main/java/com/example/cache/LRUCache.java`?

```sh
bazel query 'buildfiles(src/main/java/com/example/cache/LRUCache.java)' --output=package
```

<a name="What_is_the_build_label_for_java"></a>
#### What is the build label for src/main/java/com/example/cache/LRUCache.java?

```sh
bazel query src/main/java/com/example/cache/LRUCache.java
```

<a name="What_build_rule_contains_file_ja"></a>
#### What build rule contains file `src/main/java/com/example/cache/LRUCache.java` as a source?

```sh
fullname=$(bazel query src/main/java/com/example/cache/LRUCache.java)
bazel query "attr('srcs', $fullname, ${fullname//:*/}:*)"
```

<a name="What_package_dependencies_exist_"></a>
### What package dependencies exist ...

<a name="What_packages_does_foo_depend_on"></a>
#### What packages does `foo` depend on? (What do I need to check out to build `foo`)

```sh
bazel query 'buildfiles(deps(//foo:foo))' --output package
```

Note, `buildfiles` is required in order to correctly obtain all files
referenced by `subinclude`; see the reference manual for details.

<a name="What_packages_does_the_foo_"></a>
#### What packages does the `foo` tree depend on, excluding `foo/contrib`?

```sh
bazel query 'deps(foo/... except foo/contrib/...)' --output package
```

<a name="What_rule_dependencies_exist_"></a>
### What rule dependencies exist ...

<a name="What_genproto_rules_does_bar_"></a>
#### What genproto rules does bar depend upon?

```sh
bazel query 'kind(genproto, deps(bar/...))'
```

<a name="Find_the_definition_of_some_JNI_"></a>
#### Find the definition of some JNI (C++) library that is transitively depended upon by a Java binary rule in the servlet tree.

```sh
bazel query 'some(kind(cc_.*library, deps(kind(java_binary, src/main/java/com/example/frontend/...))))' --output location
```

<a name="_Now_find_the_definitions_of_all"></a>
##### ...Now find the definitions of all the Java binaries that depend on them

```sh
bazel query 'let jbs = kind(java_binary, src/main/java/com/example/frontend/...) in
  let cls = kind(cc_.*library, deps($jbs)) in
    $jbs intersect allpaths($jbs, $cls)'
```

<a name="What_file_dependencies_exist_"></a>
### What file dependencies exist ...

<a name="What_s_the_complete_set_of_Java_"></a>
#### What's the complete set of Java source files required to build QUX?

Source files:

```sh
bazel query 'kind("source file", deps(src/main/java/com/example/qux/...))' | grep java$
```

Generated files:

```sh
bazel query 'kind("generated file", deps(src/main/java/com/example/qux/...))' | grep java$
```

<a name="What_is_the_complete_set_of_Java"></a>
#### What is the complete set of Java source files required to build QUX's tests?

Source files:

```sh
bazel query 'kind("source file", deps(kind(".*_test rule", javatests/com/example/qux/...)))' | grep java$
```

Generated files:

```sh
bazel query 'kind("generated file", deps(kind(".*_test rule", javatests/com/example/qux/...)))' | grep java$
```

<a name="What_differences_in_dependencies"></a>
### What differences in dependencies between X and Y exist ...

<a name="What_targets_does_foo_depend_on_"></a>
#### What targets does `//foo` depend on that `//foo:foolib` does not?

```sh
bazel query 'deps(//foo) except deps(//foo:foolib)'
```

<a name="What_C_libraries_do_the_foo_test"></a>
#### What C++ libraries do the `foo` tests depend on that the `//foo` production binary does _not_ depend on?

```sh
bazel query 'kind("cc_library", deps(kind(".*test rule", foo/...)) except deps(//foo))'
```

<a name="Why_does_this_dependency_exist_"></a>
### Why does this dependency exist ...

<a name="Why_does_bar_depend_on_groups"></a>
#### Why does <code>bar</code> depend on <code>groups2</code>?

```sh
bazel query 'somepath(bar/...,groups2/...:*)'
```

Once you have the results of this query, you will often find that a single
target stands out as being an unexpected or egregious and undesirable
dependency of `bar`.  The query can then be further refined to:

<a name="Show_me_a_path_from_docker_updater"></a>
#### Show me a path from `docker/updater:updater_systest` (a `py_test`) to some `cc_library` that it depends upon:

```sh
bazel query 'let cc = kind(cc_library, deps(docker/updater:updater_systest)) in
  somepath(docker/updater:updater_systest, $cc)'
```

<a name="Why_does_library_photos_fronten"></a>
#### Why does library `//photos/frontend:lib` depend on two variants of the same library `//third_party/jpeglib` and `//third_party/jpeg`?

This query boils down to: "show me the subgraph of `//photos/frontend:lib` that
depends on both libraries".  When shown in topological order, the last element
of the result is the most likely culprit.

```
% bazel query 'allpaths(//photos/frontend:lib, //third_party/jpeglib)
                intersect
               allpaths(//photos/frontend:lib, //third_party/jpeg)'
//photos/frontend:lib
//photos/frontend:lib_impl
//photos/frontend:lib_dispatcher
//photos/frontend:icons
//photos/frontend/modules/gadgets:gadget_icon
//photos/thumbnailer:thumbnail_lib
//third_party/jpeg/img:renderer
```

<a name="What_depends_on_"></a>
### What depends on  ...

<a name="What_rules_under_bar_depend_o"></a>
#### What rules under bar depend on Y?

```sh
bazel query 'bar/... intersect allpaths(bar/..., Y)'
```

Note: `X intersect allpaths(X, Y)` is the general idiom for the query "which X
depend on Y?" If expression X is non-trivial, it may be convenient to bind a
name to it using `let` to avoid duplication.

<a name="How_do_I_break_a_dependency_"></a>
### How do I break a dependency ...

<!-- TODO find a convincing value of X to plug in here -->
<a name="What_dependency_paths_do_I_have_"></a>
#### What dependency paths do I have to break to make `bar` no longer depend on X?

To output the graph to a `png` file:

```sh
bazel query 'allpaths(bar/...,X)' --output graph | dot -Tpng > /tmp/dep.png
```

<a name="Misc_"></a>
### Misc ...

<a name="How_many_sequential_steps_are_th"></a>
#### How many sequential steps are there in the `ServletSmokeTests` build?

Unfortunately, the query language can't currently give you the longest path
from x to y, but it can find the (or rather _a_) most distant node from the
starting point, or show you the _lengths_ of the longest path from x to every
y that it depends on.  Use `maxrank`:

```sh
% bazel query 'deps(//src/test/java/com/example/servlet:ServletSmokeTests)' --output maxrank | tail -1
85 //third_party/zlib:zutil.c
```

The result indicates that there exist paths of length 85 that must occur in
order in this build.
