// Copyright 2014 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.google.devtools.build.lib.syntax;

import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.util.List;
import javax.annotation.Nullable;

/** Syntax node for an if or elif statement. */
public final class IfStatement extends Statement {

  private final TokenKind token; // IF or ELIF
  private final Expression condition;
  private final ImmutableList<Statement> thenBlock; // non-empty
  @Nullable ImmutableList<Statement> elseBlock; // non-empty if non-null; set after construction

  IfStatement(TokenKind token, Expression condition, List<Statement> thenBlock) {
    this.token = token;
    this.condition = condition;
    this.thenBlock = ImmutableList.copyOf(thenBlock);
  }

  /**
   * Reports whether this is an 'elif' statement.
   *
   * <p>An elif statement may appear only as the sole statement in the "else" block of another
   * IfStatement.
   */
  public boolean isElif() {
    return token == TokenKind.ELIF;
  }

  public Expression getCondition() {
    return condition;
  }

  public ImmutableList<Statement> getThenBlock() {
    return thenBlock;
  }

  @Nullable
  public ImmutableList<Statement> getElseBlock() {
    return elseBlock;
  }

  void setElseBlock(List<Statement> elseBlock) {
    this.elseBlock = ImmutableList.copyOf(elseBlock);
  }

  @Override
  public void prettyPrint(Appendable buffer, int indentLevel) throws IOException {
    printIndent(buffer, indentLevel);
    buffer.append(token == TokenKind.IF ? "if " : "elif ");
    condition.prettyPrint(buffer);
    buffer.append(":\n");
    printSuite(buffer, thenBlock, indentLevel);
    if (elseBlock != null) {
      if (elseBlock.size() == 1
          && elseBlock.get(0) instanceof IfStatement
          && ((IfStatement) elseBlock.get(0)).isElif()) {
        elseBlock.get(0).prettyPrint(buffer, indentLevel);
      } else {
        printIndent(buffer, indentLevel);
        buffer.append("else:\n");
        printSuite(buffer, elseBlock, indentLevel);
      }
    }
  }

  @Override
  public String toString() {
    return String.format("if %s: ...\n", condition);
  }

  @Override
  public void accept(NodeVisitor visitor) {
    visitor.visit(this);
  }

  @Override
  public Kind kind() {
    return Kind.IF;
  }
}
