// (c) Microsoft Corporation 2005-2007. 

#light


namespace Microsoft.FSharp.Collections.Tags

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics
    open System
#if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
#else
    open System.Collections.Generic
#endif

    //-------------------------------------------------------------------------
    // HashIdentity
    //-------------------------------------------------------------------------

    type StructuralHash<'a>() = 
        interface IEqualityComparer<'a> with 
          member self.GetHashCode(x) = LanguagePrimitives.GenericHash(x) 
          member self.Equals(x,y) = LanguagePrimitives.GenericEquality x y 

    type ReferenceHash<'a>() = 
        interface IEqualityComparer<'a> with 
          member self.GetHashCode(x) = LanguagePrimitives.PhysicalHash(x) 
          member self.Equals(x,y) = LanguagePrimitives.PhysicalEquality x y 

    type StructuralComparer<'a>() = 
        interface IComparer<'a> with 
          member self.Compare(x,y) = LanguagePrimitives.GenericComparison x y 

    module Optimizations = 

        type StructuralHashInt() = 
            inherit StructuralHash<int>()
            interface IEqualityComparer<int> with 
                member self.GetHashCode(x) = LanguagePrimitives.GenericHash(x) 
                member self.Equals(x,y) = LanguagePrimitives.GenericEquality x y 

        type StructuralComparerInt() = 
            inherit StructuralComparer<int>()
            interface IComparer<int> with 
                member self.Compare(x,y) = LanguagePrimitives.GenericComparison x y 


namespace Microsoft.FSharp.Collections

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics
    open System
#if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
#else
    open System.Collections.Generic
#endif

    module HashIdentity = 
        let inline Structural<'a> : Tags.StructuralHash<'a> = 
#if CLI_AT_MOST_1_1
#else
            let type_a = (type 'a) 
            // type-specialize some common cases to generate more efficient functions 
            if type_a.Equals((type int)) then unbox(box(new Tags.Optimizations.StructuralHashInt()))
            else 
#endif
                new Tags.StructuralHash<'a>()
              
        let inline Reference<'a> = new Tags.ReferenceHash<'a>() 
        let inline FromFunctions hash eq = 
            { new IEqualityComparer<'a> with 
                member self.GetHashCode(x) = hash x
                member self.Equals(x,y) = eq x y  }

        let inline FromFunction hash eq = FromFunctions hash eq
        let inline Custom hash eq = FromFunctions hash eq
            
        let inline ViaGetHashCode<'a> : IEqualityComparer<'a> = 
            { new IEqualityComparer<'a> with
                  member self.GetHashCode(o) = (box o).GetHashCode()
                  member self.Equals(obj1,obj2) = (box obj1).Equals(obj2)  }

        let inline Object<'a> : IEqualityComparer<'a> = ViaGetHashCode


    module ComparisonIdentity = 

        let inline Structural<'a> : Tags.StructuralComparer<'a> = 
#if CLI_AT_MOST_1_1
#else
            let type_a = (type 'a) 
            // type-specialize some common cases to generate more efficient functions 
            if type_a.Equals((type int)) then 
                unbox (box ( new Tags.Optimizations.StructuralComparerInt() ) )
            else 
#endif
                new Tags.StructuralComparer<'a>()
            
        let inline ViaIComparable<'a> : IComparer<'a> = 
            { new IComparer<'a> with 
                member self.Compare(x1,x2) = 
                    match box x1,box x2 with 
                    | null,null -> 0
                    | null,_ -> -1
                    | _,null -> 1
#if CLI_AT_MOST_1_1
#else
                    | (:? System.IComparable<'a> as c1),_ -> c1.CompareTo(x2)
                    | _,(:? System.IComparable<'a> as c2) -> -(c2.CompareTo(x1))
#endif
                    | (:? System.IComparable as c1),o2 -> c1.CompareTo(o2)
                    | o1,(:? System.IComparable as c2) -> -(c2.CompareTo(o1))
                    | _ -> failwith "ComparisonIdentity.ViaIComparable: the values didn't implement an IComparable interface" }

        let inline FromFunction comparer = 
            { new IComparer<'a> with 
                member self.Compare(x,y) = comparer x y }

        let inline Custom comparer = FromFunction comparer


    module Primes= 
        let primes = [| 3; 7; 11; 17; 23; 29; 37; 47; 59; 71; 89; 107; 131; 163; 197; 239; 293; 353; 431; 521; 631; 761; 919;
                        1103; 1327; 1597; 1931; 2333; 2801; 3371; 4049; 4861; 5839; 7013; 8419; 10103; 12143; 14591;
                        17519; 21023; 25229; 30293; 36353; 43627; 52361; 62851; 75431; 90523; 108631; 130363; 156437;
                        187751; 225307; 270371; 324449; 389357; 467237; 560689; 672827; 807403; 968897; 1162687; 1395263;
                        1674319; 2009191; 2411033; 2893249; 3471899; 4166287; 4999559; 5999471; 7199369 |]

        let nprimes = Array.length primes 

        let getPrime(min) =
            let rec get i prime = 
                if i < nprimes && not (prime >= min) then
                    get (i+1) primes.[i]
                elif prime >= min then prime
                else min
            get 0 0 
            
        let norm (x: int) nsize = (x &&& 0x7FFFFFFF) % nsize

    //-------------------------------------------------------------------------
    // HashChains
    //-------------------------------------------------------------------------
 
    [<CompilationRepresentation(CompilationRepresentationFlags.PermitNull)>]
    type HashChain<'a,'b> = 
        | EmptyChain // note: representation is "null"
        | Chain of { Key: 'a; 
                     Value: 'b; 
                     mutable Rest: ('a,'b) HashChain }

    //-------------------------------------------------------------------------
    // HashStats
    //-------------------------------------------------------------------------

    type HashStats = 
        { mutable ncreate:int; 
          mutable resizes: int; 
          mutable maxchains: int }
        static member Create () =  
              { ncreate=0; 
                resizes=0; 
                maxchains=0}
        member x.NumResizes= x.resizes
        member x.NumTables= x.ncreate
        member x.LongestChainLength = x.maxchains

   
    type HashMultiMap<'key,'v>  =
        { hmmOps: IEqualityComparer<'key>;
          hmmStats: HashStats;
          mutable hmmBuckets: ('key,'v) HashChain array;
          mutable hmmCount: int }
        interface IEnumerable<KeyValuePair<'key, 'v>> 
        interface System.Collections.IEnumerable 
#if CLI_AT_MOST_1_1
#else
        interface IDictionary<'key, 'v> 
        interface ICollection<KeyValuePair<'key, 'v>> 
#endif

    module HashTableOps = 

        open Primes

        let rec chainLength c = 
            match c with 
            | Chain(a,b,c) -> 1 + chainLength c
            | EmptyChain -> 0
                
        let rec foldChain f x acc =
            match x with
            | Chain(a,b,c) -> f a b (foldChain f c acc)
            | EmptyChain -> acc

        let clearBuckets (arr : ('a,'b) HashChain array) = 
            System.Array.Clear((unbox (box arr) : System.Array), 0, Array.length arr)

        let notempty = function EmptyChain -> false | Chain _ -> true

        let iterChain f chain = 
            let mutable curr = chain 
            while notempty curr do
                f curr.(Chain).0 curr.(Chain).1;
                curr <- curr.(Chain).2;

        let iteratorChain chain = 
            { let curr = ref chain 
              while notempty !curr do
                  match !curr with
                  | Chain(k,v,rest) ->
                      let res = KeyValuePair(k,v)
                      do curr :=  rest
                      yield res
                  | EmptyChain ->
                      yield! [] }

        let rec countChain acc chain = 
            match chain with 
            | Chain(a,b,c) -> countChain (acc+1) c
            | _ -> acc

        let iterBuckets f arr = 
            let len = Array.length arr 
            for i = 0 to len - 1 do 
                iterChain f arr.(i)

        let iteratorBuckets arr = 
            let len = Array.length arr 
            { for i in 0 .. len - 1 do 
                  yield! iteratorChain arr.[i] }

        let countBuckets arr = 
            Array.fold_left countChain 0 arr

        let longestChain arr = 
            let len = Array.length arr 
            let mutable mx = 0 
            let mutable longest = arr.(0) 
            for i = 0 to len - 1 do 
                let chain = arr.(i) 
                let len = chainLength chain 
                if len > mx then 
                    longest <- chain;
                mx <- max mx len;
            foldChain (fun x y acc -> (x,y) :: acc) longest []

        let addToFrontOfBucketChain hc arr nsize x y =
            let hc = norm hc nsize 
            arr.(hc) <- Chain (x,y,arr.(hc))

        let rec addChain (ops:IEqualityComparer<'a>) arr nsize curr =
            match curr with 
            | EmptyChain -> 0
            | Chain (x,y,z) -> 
                let acc = addChain ops arr nsize z 
                addToFrontOfBucketChain (ops.GetHashCode x) arr nsize x y
                acc+1

        let rec revAppendListToChain curr acc =
            match curr with 
            | [] -> acc
            | (x,y)::rest -> revAppendListToChain rest (Chain(x,y,acc))

        let rec copyChain curr acc =
            match curr with 
            | EmptyChain -> revAppendListToChain acc EmptyChain
            | Chain (x,y,z) -> copyChain z ((x,y)::acc)

        let rec chainMem (ops:IEqualityComparer<'a>) (x:'a) = function
            | Chain(a,b,c) -> ops.Equals(x, (a:'a)) || chainMem ops x c 
            | EmptyChain -> false
                
        // This relies on the fact that chains are owned by the bucket! 
        let rec chainRemove (ops:IEqualityComparer<'a>) prev (x:'a) curr = 
            match curr with 
            | Chain(a,b,next) -> 
                if ops.Equals(x, (a:'a)) then (prev.(Chain).2 <- next ; true)
                else chainRemove ops curr x next
            | EmptyChain -> false

        let rec chainFind (ops:IEqualityComparer<'a>) (x:'a) = function
            | Chain(a,b,c) -> if ops.Equals(x, (a:'a)) then b else chainFind ops x c 
            | EmptyChain -> not_found()
                
        let rec chainTryFind (ops:IEqualityComparer<'a>) (x:'a) = function
            | Chain(a,b,c) -> if ops.Equals(x, (a:'a)) then Some b else chainTryFind ops x c 
            | EmptyChain -> None
                
        let rec chainFindAll (ops:IEqualityComparer<'a>) (x:'a) y = 
            match y with 
            | EmptyChain -> []
            | Chain(x2,y,t) -> if ops.Equals(x, (x2:'a)) then y::chainFindAll ops x t else chainFindAll ops x t

        let createTable stats ops n =
            let size = getPrime (if n <= 11 then 11 else n) 
            stats.ncreate <- stats.ncreate + 1;
            {hmmOps= (ops :> IEqualityComparer<_>); hmmStats=stats; hmmBuckets = Array.create size EmptyChain; hmmCount = 0; }

        let findChain (hmmOps: IEqualityComparer<'a>) t x = 
            let arr = t.hmmBuckets 
            arr.(norm (hmmOps.GetHashCode(x)) (Array.length arr))

        let checkResize t  = 
            let arr = t.hmmBuckets 
            let osize = Array.length arr 
            (* Resize table if it looks too full. *)
            if t.hmmCount > osize then 
                let stats = t.hmmStats 
                let nsize = getPrime (t.hmmCount * 2 + 1) 
                let arr2 = Array.create nsize EmptyChain 
                for i = 0 to osize - 1 do 
                    let c = arr.(i) 
                    let cl = addChain t.hmmOps arr2 nsize c
                    if cl > stats.maxchains then 
                        stats.maxchains <- cl; 
                t.hmmBuckets <- arr2;
                stats.resizes <- stats.resizes + 1;

        let add t x y = 
            checkResize t;
            let arr2 = t.hmmBuckets 
            addToFrontOfBucketChain (t.hmmOps.GetHashCode x) arr2 (Array.length arr2) x y;
            t.hmmCount <- t.hmmCount + 1
               
        let copy t = { t with hmmBuckets= Array.map (fun c -> copyChain c []) t.hmmBuckets;
                              hmmCount=t.hmmCount }
        let find    t x = chainFind    t.hmmOps x (findChain t.hmmOps t x)
        let tryfind t x = chainTryFind t.hmmOps x (findChain t.hmmOps t x)
        let findAll t x = chainFindAll t.hmmOps x (findChain t.hmmOps t x)
        let mem     t x = chainMem     t.hmmOps x (findChain t.hmmOps t x)

        let remove t x = 
            let arr = t.hmmBuckets 
            let hc = norm (t.hmmOps.GetHashCode x) (Array.length arr) 
            match arr.(hc) with 
            | EmptyChain -> ()
            | Chain(y,_,next) as prev -> 
                if t.hmmOps.Equals(x,y) 
                then (arr.(hc) <- next;   t.hmmCount <- t.hmmCount - 1)
                elif chainRemove t.hmmOps prev x next 
                then t.hmmCount <- t.hmmCount - 1

        let replace t x y = 
            let arr = t.hmmBuckets 
            let hc = norm (t.hmmOps.GetHashCode x) (Array.length arr) 
            let next = arr.(hc) 
            let c = Chain(x,y,next) 
            arr.(hc) <- c;
            if not (chainRemove t.hmmOps c x next) then 
                t.hmmCount <- t.hmmCount + 1;
                checkResize t;

        let fold f t c = Array.fold_right (foldChain f) t.hmmBuckets c
        let iter f t = iterBuckets f t.hmmBuckets 
        let iterator t = iteratorBuckets t.hmmBuckets 
        let count t = countBuckets t.hmmBuckets 
        let clear t = 
            clearBuckets t.hmmBuckets;
            t.hmmCount <- 0

        let gen_map x f = 
            let t = createTable x.hmmStats x.hmmOps x.hmmCount 
            iter (fun k v -> add t k (f k v)) x; 
            t

    
    type HashMultiMap<'key,'v> with 
        static member Create ((hasheq: IEqualityComparer<'key>),stats,n) : HashMultiMap<'key,'v> = HashTableOps.createTable stats hasheq n 
        static member Create ((hasheq: IEqualityComparer<'key>),n) : HashMultiMap<'key,'v> = HashMultiMap.Create(hasheq ,HashStats.Create(),n)
        static member inline Create((n:int)) : HashMultiMap<'key,'v> = HashMultiMap.Create(HashIdentity.Structural,n)
        static member inline Create() : HashMultiMap<'key,'v> = HashMultiMap.Create(11)
        [<OverloadID("Create_seq")>]
        static member Create(seq : seq<'key * 'v>) : HashMultiMap<'key,'v> = 
            let t = HashMultiMap.Create(11)
            seq |> Seq.iter (fun (k,v) -> t.Add(k,v))
            t
        member x.Add(y,z) = HashTableOps.add x y z
        member x.Clear() = HashTableOps.clear x
        member x.Copy() :  HashMultiMap<'key,'v>  = HashTableOps.copy x 
        member x.Find(y) = HashTableOps.find x y
        member x.Item with get(y : 'key) = HashTableOps.find x y
                      and  set (y:'key) (z:'v) = HashTableOps.replace x y z
        member x.FindAll(y) = HashTableOps.findAll x y
        member x.Fold f acc = HashTableOps.fold f x acc
        member x.Map  f :  HashMultiMap<'key,'v>  = HashTableOps.gen_map x f
        member x.Iterate(f) =  HashTableOps.iter f x
        member x.Contains(y) = HashTableOps.mem x y
        member x.Remove(y) = HashTableOps.remove x y
        member x.Replace(y,z) = HashTableOps.replace x y z
        member x.TryFind(y) = HashTableOps.tryfind x y
        member x.Count = HashTableOps.count x 
        member x.GetLongestChain() =  HashTableOps.longestChain x.hmmBuckets 

        interface IEnumerable<KeyValuePair<'key, 'v>> with
            member s.GetEnumerator() = (HashTableOps.iterator s).GetEnumerator()
        interface System.Collections.IEnumerable with
            override s.GetEnumerator() = ((HashTableOps.iterator s).GetEnumerator() :> System.Collections.IEnumerator)
#if CLI_AT_MOST_1_1
#else
        interface IDictionary<'key, 'v> with 
            override s.Item 
                with get x = s.[x]            
                and  set x v = s.[x] <- v
            override s.Keys = ([| for kvp in s -> kvp.Key |] :> ICollection<'key>)
            override s.Values = ([| for kvp in s -> kvp.Value |] :> ICollection<'a>)
            override s.Add(k,v) = s.[k] <- v
            override s.ContainsKey(k) = s.Contains(k)
            override s.TryGetValue(k,r) = if s.Contains(k) then (r <- s.[k]; true) else false
            override s.Remove(k:'key) = let res = s.Contains(k) in s.Remove(k); res

        interface ICollection<KeyValuePair<'key, 'v>> with 
            override s.Add(x) = s.[x.Key] <- x.Value
            override s.Clear() = HashTableOps.clear s
            override s.Remove(x) = let res = s.Contains(x.Key) in HashTableOps.remove s x.Key; res
            override s.Contains(x) = s.Contains(x.Key) && s.[x.Key] = x.Value
            override s.CopyTo(arr,i) = s |> Seq.iteri (fun i x -> arr.[i] <- x)
            override s.IsReadOnly = false
            override s.Count = s.Count
#endif

    type HashTable<'key,'v> = HashMultiMap<'key,'v>


    [<Sealed>]
    type HashSet<'a> = 
        new(t) = { t = t }
        static member Make(t) : HashSet<'a> = new HashSet<_>(t)
    #if CLI_AT_LEAST_2_0
        val t: Dictionary<'a,int> 
        static member Create((hasheq: (IEqualityComparer<'a>)),(stats:HashStats),(n:int)) : HashSet<'a> = HashSet.Make(new Dictionary<_,_>(n,hasheq))
        static member Create((hasheq: (IEqualityComparer<'a>)),(n:int)) : HashSet<'a> = HashSet.Make(new Dictionary<_,_>(n,hasheq))
        member x.Add(y)    = x.t.[y] <- 0
        member x.Clear() = x.t.Clear()
        member x.Copy() : HashSet<'a>  = let t = new Dictionary<_,_>(x.t.Count,x.t.Comparer) in x.t |> Seq.iter (fun kvp -> t.[kvp.Key] <- 0); new HashSet<_>(t)
        member x.Fold f acc = x.t |> Seq.fold (fun acc kvp -> f kvp.Key acc) acc
        member x.Map f : HashSet<'a>  = let t = new Dictionary<_,_>(x.t.Count,x.t.Comparer) in x.t |> Seq.iter (fun kvp -> t.[f kvp.Key] <- 0); new HashSet<_>(t)
        member x.Iterate(f) =  x.t |> Seq.iter (fun kvp -> f kvp.Key)
        member x.Contains(y) = x.t.ContainsKey(y)
        member x.Remove(y) = x.t.Remove(y) |> ignore
        member x.Count = x.t.Count
        member x.GetLongestChain() = ([] : 'a list)
        interface IEnumerable<'a> with
          member x.GetEnumerator() = x.t.Keys.GetEnumerator() :> System.Collections.Generic.IEnumerator<_>
        interface System.Collections.IEnumerable with
          member x.GetEnumerator() = x.t.Keys.GetEnumerator()  :> System.Collections.IEnumerator 
    #else
        val t: HashMultiMap<'a,int>
        static member Create(hasheq,n) : HashSet<'a> = new HashSet<_>(HashMultiMap.Create(hasheq,n))
        static member Create((hasheq: IEqualityComparer<'a>),stats,n) : HashSet<'a> = new HashSet<_>(HashMultiMap.Create(hasheq,stats,n))
        member x.Add(y)    = x.t.Replace(y,0);
        member x.Clear() = x.t.Clear()
        member x.Copy() = HashSet.Make(x.t.Copy())
        member x.Fold f acc = x.t.Fold(fun y _ acc -> f y acc) acc
        member x.Map f = let t = HashMultiMap.Create(x.t.hmmOps,x.t.hmmStats,x.t.hmmCount) in t.Iterate(fun x _ -> t.Replace(f x,0)); HashSet.Make(t)
        member x.Iterate(f) =  x.t.Iterate(fun x _ -> f x)
        member x.Contains(y) = x.t.Contains(y)
        member x.Remove(y) = x.t.Remove(y)
        member x.Count = x.t.Count
        member x.GetLongestChain() = List.map fst (x.t.GetLongestChain())
    #endif
        static member inline Create(n:int) : HashSet<'a>  = HashSet.Create((HashIdentity.Structural :> IEqualityComparer<_>),n)
        static member inline Create() : HashSet<'a> = HashSet.Create(11)
        [<OverloadID("Create_seq")>]
        static member Create(seq:seq<'a>) : HashSet<'a> = 
            let t = HashSet.Create(1)
            seq |> Seq.iter t.Add
            t


    type CHashTable<'a,'b> = HashTable<'a,'b>
    type CHashSet<'a> = HashSet<'a>


    (* From "Algorithms", Cormen, Leiserson & Rivest p 252. *)

    [<CompilationRepresentation(CompilationRepresentationFlags.PermitNull)>]
    type 'a SetTree = 
        | SetEmpty                                          // height = 0   
        | SetNode of 'a * 'a SetTree *  'a SetTree * int    // height = int 
        | SetOne  of 'a                                     // height = 1   
        // OPTIMIZATION: store SetNode(k,SetEmpty,SetEmpty,1) --->  SetOne(k,SetEmpty) 

    type Set<'a> = 
        { cf: IComparer<'a>;
          tree: SetTree<'a> }
        interface System.IComparable 
        //interface IStructuralHash 
#if CLI_AT_MOST_1_1
#else
        interface ICollection<'a> 
#endif
        interface IEnumerable<'a> 
        interface System.Collections.IEnumerable 


    module RawSetOps = 
        let empty = SetEmpty

        let height t = 
            match t with 
            | SetEmpty -> 0
            | SetOne _ -> 1
            | SetNode (_,_,_,h) -> h

        let rec checkInvariant t =
            // A good sanity check, loss of balance can hit perf 
            match t with 
            | SetEmpty -> true
            | SetOne _ -> true
            | SetNode (k,t1,t2,h) ->
                let h1 = height t1 in
                let h2 = height t2 in
                (-2 <= (h1 - h2) && (h1 - h2) <= 2) && checkInvariant t1 && checkInvariant t2

        let tolerance = 2

        let mk l k r = 
            match l,r with 
            | SetEmpty,SetEmpty -> SetOne (k)
            | _ -> 
              let hl = height l in 
              let hr = height r in 
              let m = if hl < hr then hr else hl in 
              SetNode(k,l,r,m+1)

        let rebalance t1 k t2 =
            let t1h = height t1 
            if  height t2 > t1h + tolerance then // right is heavier than left 
                match t2 with 
                | SetNode(t2k,t2l,t2r,t2h) -> 
                    // one of the nodes must have height > height t1 + 1 
                    if height t2l > t1h + 1 then  // balance left: combination 
                        match t2l with 
                        | SetNode(t2lk,t2ll,t2lr,t2lh) ->
                            mk (mk t1 k t2ll) t2lk (mk t2lr t2k t2r) 
                        | _ -> failwith "rebalance"
                    else // rotate left 
                        mk (mk t1 k t2l) t2k t2r
                | _ -> failwith "rebalance"
            else
                let t2h = height t2 
                if  t1h > t2h + tolerance then // left is heavier than right 
                    match t1 with 
                    | SetNode(t1k,t1l,t1r,t1h) -> 
                        // one of the nodes must have height > height t2 + 1 
                        if height t1r > t2h + 1 then 
                            // balance right: combination 
                            match t1r with 
                            | SetNode(t1rk,t1rl,t1rr,t1rh) ->
                                mk (mk t1l t1k t1rl) t1rk (mk t1rr k t2)
                            | _ -> failwith "rebalance"
                        else
                            mk t1l t1k (mk t1r k t2)
                    | _ -> failwith "rebalance"
                else mk t1 k t2

        let rec add (cf: IComparer<_>) k t = 
            match t with 
            | SetNode (k2,l,r,h) -> 
                let c = cf.Compare(k,k2) 
                if   c < 0 then rebalance (add cf k l) k2 r
                elif c = 0 then t
                else            rebalance l k2 (add cf k r)
            | SetOne(k2) -> 
                // nb. no check for rebalance needed for small trees, also be sure to reuse node already allocated 
                let c = cf.Compare(k,k2) 
                if c < 0   then SetNode (k,SetEmpty,t,2)
                elif c = 0 then t
                else            SetNode (k,t,SetEmpty,2)                  
            | SetEmpty -> SetOne(k)

        let rec balance cf t1 k t2 =
            // Given t1 < k < t2 where t1 and t2 are "balanced",
            // return a balanced tree for <t1,k,t2>.
            // Recall: balance means subtrees heights differ by at most "tolerance"
            match t1,t2 with
            | SetEmpty,t2  -> add cf k t2 // drop t1 = empty 
            | t1,SetEmpty  -> add cf k t1 // drop t2 = empty 
            | SetOne k1,t2 -> add cf k (add cf k1 t2)
            | t1,SetOne k2 -> add cf k (add cf k2 t1)
            | SetNode(k1,t11,t12,h1),SetNode(k2,t21,t22,h2) ->
                // Have:  (t11 < k1 < t12) < k < (t21 < k2 < t22)
                // Either (a) h1,h2 differ by at most 2 - no rebalance needed.
                //        (b) h1 too small, i.e. h1+2 < h2
                //        (c) h2 too small, i.e. h2+2 < h1 
                if   h1+tolerance < h2 then
                    // case: b, h1 too small 
                    // push t1 into low side of t2, may increase height by 1 so rebalance 
                    rebalance (balance cf t1 k t21) k2 t22
                elif h2+tolerance < h1 then
                    // case: c, h2 too small 
                    // push t2 into high side of t1, may increase height by 1 so rebalance 
                    rebalance t11 k1 (balance cf t12 k t2)
                else
                    // case: a, h1 and h2 meet balance requirement 
                    mk t1 k t2

        let rec split (cf : IComparer<_>) pivot t =
            // Given a pivot and a set t
            // Return { x in t s.t. x < pivot }, pivot in t? , { x in t s.t. x > pivot } 
            match t with
            | SetNode(k1,t11,t12,h1) ->
                let c = cf.Compare(pivot,k1)
                if   c < 0 then // pivot t1 
                    let t11_lo,havePivot,t11_hi = split cf pivot t11
                    t11_lo,havePivot,balance cf t11_hi k1 t12
                elif c = 0 then // pivot is k1 
                    t11,true,t12
                else            // pivot t2 
                    let t12_lo,havePivot,t12_hi = split cf pivot t12
                    balance cf t11 k1 t12_lo,havePivot,t12_hi
            | SetOne k1 ->
                let c = cf.Compare(k1,pivot)
                if   c < 0 then t       ,false,SetEmpty // singleton under pivot 
                elif c = 0 then SetEmpty,true ,SetEmpty // singleton is    pivot 
                else            SetEmpty,false,t        // singleton over  pivot 
            | SetEmpty  -> 
                SetEmpty,false,SetEmpty
        
        let rec spliceOutSuccessor t = 
            match t with 
            | SetEmpty -> failwith "internal error: Map.splice_out_succ_or_pred"
            | SetOne (k2) -> k2,empty
            | SetNode (k2,l,r,_) ->
                match l with 
                | SetEmpty -> k2,r
                | _ -> let k3,l' = spliceOutSuccessor l in k3,mk l' k2 r

        let rec remove (cf: IComparer<_>) k t = 
            match t with 
            | SetEmpty -> t
            | SetOne (k2) -> 
                let c = cf.Compare(k,k2) 
                if   c = 0 then empty
                else            t
            | SetNode (k2,l,r,_) -> 
                let c = cf.Compare(k,k2) 
                if   c < 0 then rebalance (remove cf k l) k2 r
                elif c = 0 then 
                  match l,r with 
                  | SetEmpty,_ -> r
                  | _,SetEmpty -> l
                  | _ -> 
                      let sk,r' = spliceOutSuccessor r 
                      mk l sk r'
                else rebalance l k2 (remove cf k r) 

        let rec mem (cf: IComparer<_>) k t = 
            match t with 
            | SetNode(k2,l,r,_) -> 
                let c = cf.Compare(k,k2) 
                if   c < 0 then mem cf k l
                elif c = 0 then true
                else mem cf k r
            | SetOne(k2) -> (cf.Compare(k,k2) = 0)
            | SetEmpty -> false

        let rec iter f t = 
            match t with 
            | SetNode(k2,l,r,_) -> iter f l; f k2; iter f r
            | SetOne(k2) -> f k2
            | SetEmpty -> ()            

        let rec fold f m x = 
            match m with 
            | SetNode(k,l,r,h) -> fold f r (f k (fold f l x))
            | SetOne(k) -> f k x
            | SetEmpty -> x                

        let rec foldr f m x = 
            match m with 
            | SetNode(k,l,r,h) -> foldr f l (f k (foldr f r x))
            | SetOne(k) -> f k x
            | SetEmpty -> x

        let rec for_all f m = 
            match m with 
            | SetNode(k2,l,r,h) -> f k2 && for_all f l && for_all f r
            | SetOne(k2) -> f k2
            | SetEmpty -> true          

        let rec exists f m = 
            match m with 
            | SetNode(k2,l,r,h) -> f k2 || exists f l || exists f r
            | SetOne(k2) -> f k2
            | SetEmpty -> false         

        let is_empty m = match m with  | SetEmpty -> true | _ -> false

        let subset (cf: IComparer<_>) a b  = for_all (fun x -> mem cf x b) a

        let rec elementsAux m acc = 
            match m with 
            | SetNode(k2,l,r,_) -> k2 :: (elementsAux l (elementsAux r acc))
            | SetOne(k2) -> k2 :: acc
            | SetEmpty -> acc                

        let elements a  = elementsAux a []

        let rec filterAux (cf: IComparer<_>) f s acc = 
            match s with 
            | SetNode(k,l,r,_) -> 
                let acc = if f k then add cf k acc else acc 
                filterAux cf f l (filterAux cf f r acc)
            | SetOne(k) -> if f k then add cf k acc else acc
            | SetEmpty -> acc           

        let filter (cf: IComparer<_>) f s = filterAux cf f s empty

        let rec diffAux (cf: IComparer<_>) m acc = 
            match m with 
            | SetNode(k,l,r,_) -> diffAux cf l (diffAux cf r (remove cf k acc))
            | SetOne(k) -> remove cf k acc
            | SetEmpty -> acc           

        let diff (cf: IComparer<_>) a b = diffAux cf b a

        let rec cardinalAux s acc = 
            match s with 
            | SetNode(k,l,r,_) -> cardinalAux l (cardinalAux r (acc+1))
            | SetOne(k) -> acc+1
            | SetEmpty -> acc           

        let cardinal s = cardinalAux s 0
        let size s = cardinal s 

        let rec union (cf: IComparer<_>) t1 t2 =
            // Perf: tried bruteForce for low heights, but nothing significant 
            match t1,t2 with               
            | SetNode(k1,t11,t12,h1),SetNode(k2,t21,t22,h2) -> // (t11 < k < t12) AND (t21 < k2 < t22) 
                // Divide and Quonquer:
                //   Suppose t1 is largest.
                //   Split t2 using pivot k1 into lo and hi.
                //   Union disjoint subproblems and then combine. 
                if h1 > h2 then
                  let lo,_,hi = split cf k1 t2 in
                  balance cf (union cf t11 lo) k1 (union cf t12 hi)
                else
                  let lo,_,hi = split cf k2 t1 in
                  balance cf (union cf t21 lo) k2 (union cf t22 hi)
            | SetEmpty,t -> t
            | t,SetEmpty -> t
            | SetOne k1,t2 -> add cf k1 t2
            | t1,SetOne k2 -> add cf k2 t1

        let rec intersectionAux (cf: IComparer<_>) b m acc = 
            match m with 
            | SetNode(k,l,r,_) -> 
                let acc = intersectionAux cf b r acc 
                let acc = if mem cf k b then add cf k acc else acc 
                intersectionAux cf b l acc
            | SetOne(k) -> 
                if mem cf k b then add cf k acc else acc
            | SetEmpty -> acc

        let intersection (cf: IComparer<_>) a b = intersectionAux cf b a empty

        let partition1 (cf: IComparer<_>) f k (acc1,acc2) = if f k then (add cf k acc1,acc2) else (acc1,add cf k acc2) 
        
        let rec partitionAux (cf: IComparer<_>) f s acc = 
            match s with 
            | SetNode(k,l,r,_) -> 
                let acc = partitionAux cf f r acc 
                let acc = partition1 cf f k acc
                partitionAux cf f l acc
            | SetOne(k) -> partition1 cf f k acc
            | SetEmpty -> acc           

        let partition cf f s = partitionAux cf f s (empty,empty)

        // It's easier to get many less-important algorithms right using this active pattern
        let (|MatchSetNode|MatchSetEmpty|) s = 
            match s with 
            | SetNode(k2,l,r,_) -> MatchSetNode(k2,l,r)
            | SetOne(k2) -> MatchSetNode(k2,SetEmpty,SetEmpty)
            | SetEmpty -> MatchSetEmpty
        
        let rec nextElemCont (cf: IComparer<_>) k s cont = 
            match s with 
            | MatchSetNode(k2,l,r) -> 
                let c = cf.Compare(k,k2) 
                if   c < 0 then nextElemCont cf k l (function None -> cont(Some(k2)) | res -> res)
                elif c = 0 then cont(minimumElementOpt r) 
                else nextElemCont cf k r cont
            | MatchSetEmpty -> cont(None)

        and nextElem cf k s = nextElemCont cf k s (fun res -> res)
        
        and prevElemCont (cf: IComparer<_>) k s cont = 
            match s with 
            | MatchSetNode(k2,l,r) -> 
                let c = cf.Compare(k,k2) 
                if   c > 0 then prevElemCont cf k r (function None -> cont(Some(k2)) | res -> res)
                elif c = 0 then cont(maximumElementOpt r) 
                else prevElemCont cf k l cont
            | MatchSetEmpty -> cont(None)

        and prevElem cf k s = prevElemCont cf k s (fun res -> res)
        
        and minimumElementAux s n = 
            match s with 
            | SetNode(k,l,r,_) -> minimumElementAux l k
            | SetOne(k) -> k
            | SetEmpty -> n

        and minimumElementOpt s = 
            match s with 
            | SetNode(k,l,r,_) -> Some(minimumElementAux l k)
            | SetOne(k) -> Some k
            | SetEmpty -> None

        and maximumElementAux s n = 
            match s with 
            | SetNode(k,l,r,_) -> maximumElementAux r k
            | SetOne(k) -> k
            | SetEmpty -> n             

        and maximumElementOpt s = 
            match s with 
            | SetNode(k,l,r,_) -> Some(maximumElementAux r k)
            | SetOne(k) -> Some(k)
            | SetEmpty -> None

        let minimumElement s = 
            match minimumElementOpt s with 
            | Some(k) -> k
            | None -> failwith "minimumElement"            

        let maximumElement s = 
            match maximumElementOpt s with 
            | Some(k) -> k
            | None -> failwith "maximumElement"


        //--------------------------------------------------------------------------
        // Imperative left-to-right iterators.
        //--------------------------------------------------------------------------

        type 'a iterator = { mutable stack: 'a SetTree list;  // invariant: always collapseLHS result 
                             mutable started : bool           // true when MoveNext has been called   
                           }

        // collapseLHS:
        // a) Always returns either [] or a list starting with SetOne.
        // b) The "fringe" of the set stack is unchanged.
        let rec collapseLHS stack =
            match stack with
            | []                       -> []
            | SetEmpty         :: rest -> collapseLHS rest
            | SetOne k         :: rest -> stack
            | SetNode(k,l,r,h) :: rest -> collapseLHS (l :: SetOne k :: r :: rest)
          
        let mkIterator s = { stack = collapseLHS [s]; started = false }

        let not_started() = raise (new System.InvalidOperationException("Enumeration has not started. Call MoveNext."))
        let already_finished() = raise (new System.InvalidOperationException("Enumeration already finished."))

        let current i =
            if i.started then
                match i.stack with
                  | SetOne k :: _ -> k
                  | []            -> already_finished()
                  | _             -> failwith "Please report error: Set iterator, unexpected stack for current"
            else
                not_started()

        let rec moveNext i =
            if i.started then
                match i.stack with
                  | SetOne k :: rest -> ( i.stack <- collapseLHS rest;
                                          i.stack <> []
                                        )
                  | [] -> false
                  | _ -> failwith "Please report error: Set iterator, unexpected stack for moveNext"
            else
                i.started <- true;  // The first call to MoveNext "starts" the enumeration.
                i.stack <> []

        let mkIEnumerator s = 
            let i = ref (mkIterator s) 
            { new IEnumerator<_> with 
                  member x.Current = current !i
              interface System.Collections.IEnumerator with 
                  member x.Current = box (current !i)
                  member x.MoveNext() = moveNext !i
                  member x.Reset() = i :=  mkIterator s
              interface System.IDisposable with 
                  member x.Dispose() = () }

(*
        let seqMinToMax s = 
            { new IEnumerable<_> with 
                  member e.GetEnumerator() = mkIEnumerator s
              interface System.Collections.IEnumerable with
                  member e.GetEnumerator() = (mkIEnumerator s :> System.Collections.IEnumerator) }



        let rec seqBetweenCont (cf: IComparer<_>) kStart kEnd s cont = 
            match s with 
            | MatchSetNode(k2,l,r) -> 
                let cStart = cf.Compare(kStart,k2) 
                let cEnd = cf.Compare(kEnd,k2) 
                if   cStart = 0 && cEnd = 0 then cont(Seq.singleton(k2))
                elif cEnd < 0 or cStart > 0 then cont(Seq.empty)
                elif cStart = 0 then cont(Seq.append(Seq.singleton(k2)) (seqUntil cf kEnd s))
                elif cEnd = 0 then cont(Seq.append(seqFrom cf kStart s) (Seq.singleton(k2)))
                else cont(seqMinToMax s)
            | MatchSetEmpty -> cont(Seq.empty)

        let prevElem cf k s = prevElemCont cf k s (fun res -> res)
*)
                   
        //--------------------------------------------------------------------------
        // Set comparison.  This can be expensive.
        //--------------------------------------------------------------------------

        let rec compareStacks (cf: IComparer<_>) l1 l2 =
            match l1,l2 with 
            | [],[] ->  0
            | [],_  -> -1
            | _ ,[] ->  1
            | (SetEmpty  _ :: t1),(SetEmpty    :: t2) -> compareStacks cf t1 t2
            | (SetOne(n1k) :: t1),(SetOne(n2k) :: t2) -> 
                 let c = cf.Compare(n1k,n2k) 
                 if c <> 0 then c else compareStacks cf t1 t2
            | (SetOne(n1k) :: t1),(SetNode(n2k,SetEmpty,n2r,_) :: t2) -> 
                 let c = cf.Compare(n1k,n2k) 
                 if c <> 0 then c else compareStacks cf (empty :: t1) (n2r :: t2)
            | (SetNode(n1k,(SetEmpty as emp),n1r,_) :: t1),(SetOne(n2k) :: t2) -> 
                 let c = cf.Compare(n1k,n2k) 
                 if c <> 0 then c else compareStacks cf (n1r :: t1) (emp :: t2)
            | (SetNode(n1k,(SetEmpty as emp),n1r,_) :: t1),(SetNode(n2k,SetEmpty,n2r,_) :: t2) -> 
                 let c = cf.Compare(n1k,n2k) 
                 if c <> 0 then c else compareStacks cf (n1r :: t1) (n2r :: t2)
            | (SetOne(n1k) :: t1),_ -> 
                compareStacks cf (empty :: SetOne(n1k) :: t1) l2
            | (SetNode(n1k,n1l,n1r,_) :: t1),_ -> 
                compareStacks cf (n1l :: SetNode(n1k,empty,n1r,0) :: t1) l2
            | _,(SetOne(n2k) :: t2) -> 
                compareStacks cf l1 (empty :: SetOne(n2k) :: t2)
            | _,(SetNode(n2k,n2l,n2r,_) :: t2) -> 
                compareStacks cf l1 (n2l :: SetNode(n2k,empty,n2r,0) :: t2)
                
        let compare cf s1 s2 = 
            match s1,s2 with 
            | SetEmpty,SetEmpty -> 0
            | SetEmpty,_ -> -1
            | _,SetEmpty -> 1
            | _ -> compareStacks cf [s1] [s2]

        let choose s = minimumElement s

        let to_list s = foldr (fun x y -> x::y) s  []

        let copyToArray s arr i =
            let j = ref i 
            iter (fun x -> arr.(!j) <- x; j := !j + 1) s

        let to_array s = 
            let n = (cardinal s) 
            let res = Array.zero_create n 
            copyToArray s res 0;
            res



        let rec mkFromEnumerator cf acc (e : IEnumerator<_>) = 
          if e.MoveNext() then 
            mkFromEnumerator cf (add cf e.Current acc) e
          else acc
          
        let of_seq cf (c : #IEnumerable<_>) =
          mkFromEnumerator cf empty (c.GetEnumerator()) 

        let of_array cf l = Array.fold_left (fun acc k -> add cf k acc) empty l    

    open RawSetOps

    module SetOps = 
        let baked cf t =     { cf=(cf :> IComparer<_>) ; tree=t }
        let refresh s t =    { cf=s.cf; tree=t }
        let fresh cf =       baked cf empty

    open SetOps


    type Set<'a> with 
        // REVIEW: we can use .NET generics per-instantiation static fields to avoid allocating a new object for each empty
        // list (it will just be a quick lookup into a table of type-instantiation-indexed static fields).
        static member Empty() : Set<'a> = (fresh ComparisonIdentity.Structural : Set<'a>)
        member s.Add(x) : Set<'a> = refresh s (add s.cf x s.tree)
        member s.Remove(x) : Set<'a> = refresh s (remove s.cf x s.tree)
        member s.Size = size s.tree
        member s.Count = size s.tree
        member s.Contains(x) = mem s.cf  x s.tree
        member s.Iterate(x) = iter  x s.tree
        member s.Fold f x  = fold f s.tree x
        member s.CheckBalanceInvariant = checkInvariant s.tree // diagnostics...
        member s.IsEmpty  = is_empty s.tree
        member s.Partition f  : Set<'a> *  Set<'a> = 
            match s.tree with 
            | SetEmpty -> s,s
            | _ -> let t1,t2 = partition s.cf f s.tree in refresh s t1, refresh s t2
        member s.Filter f  : Set<'a> = 
            match s.tree with 
            | SetEmpty -> s
            | _ -> filter s.cf f s.tree |> refresh s
        member s.Map f  : Set<'b> = 
            let cf = (ComparisonIdentity.Structural :> IComparer<_>)
            {cf = cf; 
             tree=fold (fun k acc -> add cf (f k) acc) s.tree empty}
        member s.Exists f = exists f s.tree
        member s.ForAll f = for_all f s.tree
        static member (-) (a: Set<'a>, b: Set<'a>) = Set.Difference(a,b)
        static member (+) (a: Set<'a>, b: Set<'a>) = Set.Union(a,b)
        static member Intersection(a: Set<'a>, b: Set<'a>) : Set<'a>  = 
            match b.tree with 
            | SetEmpty -> b  (* A INTER 0 = 0 *)
            | _ -> 
            match a.tree with 
            | SetEmpty -> a (* 0 INTER B = 0 *)
            | _ -> intersection a.cf  a.tree b.tree |> refresh a
           
        static member Union(a: Set<'a>, b: Set<'a>) : Set<'a>  = 
            match b.tree with 
            | SetEmpty -> a  (* A U 0 = A *)
            | _ -> 
            match a.tree with 
            | SetEmpty -> b  (* 0 U B = B *)
            | _ -> union a.cf  a.tree b.tree |> refresh a

        static member Union(sets:seq<Set<'a>>) : Set<'a>  = 
            Seq.fold (fun s1 s2 -> Set.Union(s1,s2)) (Set.Empty()) sets

        static member Intersection(sets:seq<Set<'a>>) : Set<'a>  = 
            Seq.fold1 (fun s1 s2 -> Set.Intersection(s1,s2)) sets

        static member Difference(a: Set<'a>, b: Set<'a>) : Set<'a>  = 
            match a.tree with 
            | SetEmpty -> a (* 0 - B = 0 *)
            | _ -> 
            match b.tree with 
            | SetEmpty -> a (* A - 0 = A *)
            | _ -> diff a.cf  a.tree b.tree |> refresh a
        static member Equality(a: Set<'a>, b: Set<'a>) = (RawSetOps.compare a.cf  a.tree b.tree = 0)
        static member Compare(a: Set<'a>, b: Set<'a>) = RawSetOps.compare a.cf  a.tree b.tree

        member x.Choose = choose x.tree
        member x.MinimumElement = minimumElement x.tree
        member x.MaximumElement = maximumElement x.tree
        member x.GetNextElement(e) = nextElem x.cf e x.tree
        member x.GetPreviousElement(e) = prevElem x.cf  e x.tree

        member x.IsSubsetOf(y: Set<'a>) = subset x.cf x.tree y.tree 
        member x.IsSupersetOf(y: Set<'a>) = subset x.cf y.tree x.tree
        member x.ToList () = to_list x.tree
        member x.ToArray () = to_array x.tree

        interface System.IComparable with 
            member this.CompareTo(that: obj) = RawSetOps.compare this.cf this.tree ((that :?> Set<'a>).tree)
        override this.Equals(that:obj) = ((this :> System.IComparable).CompareTo(that) = 0)

          
#if CLI_AT_LEAST_2_0
        interface ICollection<'a> with 
            member s.Add(x) = raise (new System.NotSupportedException("ReadOnlyCollection"))
            member s.Clear() = raise (new System.NotSupportedException("ReadOnlyCollection"))
            member s.Remove(x) = raise (new System.NotSupportedException("ReadOnlyCollection"))
            member s.Contains(x) = mem s.cf x s.tree
            member s.CopyTo(arr,i) = copyToArray s.tree arr i
            member s.get_IsReadOnly() = true
            member s.get_Count() = cardinal s.tree  
#endif
        interface IEnumerable<'a> with
            member s.GetEnumerator() = mkIEnumerator s.tree
        interface System.Collections.IEnumerable with
            override s.GetEnumerator() = (mkIEnumerator s.tree :> System.Collections.IEnumerator)

    type Set<'a> with
        static member Singleton(x) : Set<'a> = (Set.Empty()).Add(x)
        static member Create(l : seq<'a>) : Set<'a> = 
            let cf = ComparisonIdentity.Structural  
            baked cf (of_seq (cf :> IComparer<'a>) l)
          
        static member Create() : Set<'a> = Set.Empty()
        static member FromArray(arr : 'a array) : Set<'a> = 
            let cf = ComparisonIdentity.Structural  
            baked cf (of_array (cf :> IComparer<'a>) arr)

    [<CompilationRepresentation(CompilationRepresentationFlags.PermitNull)>]
    type MapTree<'key,'a> = 
        | MapEmpty 
        | MapOne of 'key * 'a
        // TODO: performance rumour has it that the data held in this node should be
        // exactly one cache line. It is currently ~7 words. Thus it might be better to
        // move to a n-way tree.
        | MapNode of 'key * 'a * MapTree<'key,'a> *  MapTree<'key,'a> * int

#if CLI_AT_MOST_1_1
#else
    [<System.Diagnostics.DebuggerDisplay("#Entries = {Size}") >]
    [<System.Diagnostics.DebuggerTypeProxy((type MapDebugView<_,_>))  >]
#endif
    type Map<'key,'a>  =  
            { cf: IComparer<'key>; tree: MapTree<'key,'a> }
            interface System.IComparable
            interface IEnumerable<KeyValuePair<'key, 'a>>         
            interface System.Collections.IEnumerable 
#if CLI_AT_MOST_1_1
#else
            interface ICollection<KeyValuePair<'key, 'a>> 
            interface IDictionary<'key, 'a>         
    and 
      [<Sealed>]
      MapDebugView<'key,'a>  =  
         val v: Map<'key,'a>
         new(x) = { v = x }     
#endif

    module RawMapOps = 

        let empty = MapEmpty 

        let height  = function
          | MapEmpty -> 0
          | MapOne _ -> 1
          | MapNode(_,_,_,_,h) -> h

        let is_empty m = 
            match m with 
            | MapEmpty -> true
            | _ -> false

        let mk l k v r = 
            match l,r with 
            | MapEmpty,MapEmpty -> MapOne(k,v)
            | _ -> 
                let hl = height l 
                let hr = height r 
                let m = if hl < hr then hr else hl 
                MapNode(k,v,l,r,m+1)

        let rebalance t1 k v t2 =
            let t1h = height t1 
            if  height t2 > t1h + 2 then (* right is heavier than left *)
                match t2 with 
                | MapNode(t2k,t2v,t2l,t2r,t2h) -> 
                   (* one of the nodes must have height > height t1 + 1 *)
                   if height t2l > t1h + 1 then  (* balance left: combination *)
                     match t2l with 
                     | MapNode(t2lk,t2lv,t2ll,t2lr,t2lh) ->
                        mk (mk t1 k v t2ll) t2lk t2lv (mk t2lr t2k t2v t2r) 
                     | _ -> failwith "rebalance"
                   else (* rotate left *)
                     mk (mk t1 k v t2l) t2k t2v t2r
                | _ -> failwith "rebalance"
            else
                let t2h = height t2 
                if  t1h > t2h + 2 then (* left is heavier than right *)
                  match t1 with 
                  | MapNode(t1k,t1v,t1l,t1r,t1h) -> 
                    (* one of the nodes must have height > height t2 + 1 *)
                      if height t1r > t2h + 1 then 
                      (* balance right: combination *)
                        match t1r with 
                        | MapNode(t1rk,t1rv,t1rl,t1rr,t1rh) ->
                            mk (mk t1l t1k t1v t1rl) t1rk t1rv (mk t1rr k v t2)
                        | _ -> failwith "rebalance"
                      else
                        mk t1l t1k t1v (mk t1r k v t2)
                  | _ -> failwith "rebalance"
                else mk t1 k v t2

        let rec sizeAux acc m = 
            match m with  
            | MapEmpty -> acc
            | MapOne _ -> acc + 1
            | MapNode(k2,v2,l,r,h) -> sizeAux (sizeAux (acc+1) l) r 

        let size x = sizeAux 0 x

        let rec add (cf: IComparer<_>) k v m = 
            match m with 
            | MapEmpty -> MapOne(k,v)
            | MapOne(k2,v2) -> 
                let c = cf.Compare(k,k2) 
                if c < 0   then MapNode (k,v,MapEmpty,m,2)
                elif c = 0 then MapOne(k,v)
                else            MapNode (k,v,m,MapEmpty,2)
            | MapNode(k2,v2,l,r,h) -> 
                let c = cf.Compare(k,k2) 
                if c < 0 then rebalance (add cf k v l) k2 v2 r
                elif c = 0 then MapNode(k,v,l,r,h)
                else rebalance l k2 v2 (add cf k v r) 

        let rec find (cf: IComparer<_>) k m = 
            match m with 
            | MapEmpty -> not_found()
            | MapOne(k2,v2) -> 
                let c = cf.Compare(k,k2) 
                if c = 0 then v2
                else not_found()
            | MapNode(k2,v2,l,r,_) -> 
                let c = cf.Compare(k,k2) 
                if c < 0 then find cf k l
                elif c = 0 then v2
                else find cf k r

        let rec tryfind (cf: IComparer<_>) k m = 
            match m with 
            | MapEmpty -> None
            | MapOne(k2,v2) -> 
                let c = cf.Compare(k,k2) 
                if c = 0 then Some v2
                else None
            | MapNode(k2,v2,l,r,_) -> 
                let c = cf.Compare(k,k2) 
                if c < 0 then tryfind cf k l
                elif c = 0 then Some v2
                else tryfind cf k r

        let partition1 (cf: IComparer<_>) f k v (acc1,acc2) = 
            if f k v then (add cf k v acc1,acc2) else (acc1,add cf k v acc2) 
        
        let rec partitionAux (cf: IComparer<_>) f s acc = 
            match s with 
            | MapEmpty -> acc
            | MapOne(k,v) -> partition1 cf f k v acc
            | MapNode(k,v,l,r,_) -> 
                let acc = partitionAux cf f r acc 
                let acc = partition1 cf f k v acc
                partitionAux cf f l acc

        let partition (cf: IComparer<_>) f s = partitionAux cf f s (empty,empty)

        let filter1 (cf: IComparer<_>) f k v acc = if f k v then add cf k v acc else acc 

        let rec filterAux (cf: IComparer<_>) f s acc = 
            match s with 
            | MapEmpty -> acc
            | MapOne(k,v) -> filter1 cf f k v acc
            | MapNode(k,v,l,r,_) ->
                let acc = filterAux cf f l acc
                let acc = filter1 cf f k v acc
                filterAux cf f r acc

        let filter (cf: IComparer<_>) f s = filterAux cf f s empty

        let rec spliceOutSuccessor m = 
            match m with 
            | MapEmpty -> failwith "internal error: Map.splice_out_succ_or_pred"
            | MapOne(k2,v2) -> k2,v2,MapEmpty
            | MapNode(k2,v2,l,r,_) ->
                match l with 
                | MapEmpty -> k2,v2,r
                | _ -> let k3,v3,l' = spliceOutSuccessor l in k3,v3,mk l' k2 v2 r

        let rec remove (cf: IComparer<_>) k m = 
            match m with 
            | MapEmpty -> empty
            | MapOne(k2,v2) -> 
                let c = cf.Compare(k,k2) 
                if c = 0 then MapEmpty else m
            | MapNode(k2,v2,l,r,_) -> 
                let c = cf.Compare(k,k2) 
                if c < 0 then rebalance (remove cf k l) k2 v2 r
                elif c = 0 then 
                  match l,r with 
                  | MapEmpty,_ -> r
                  | _,MapEmpty -> l
                  | _ -> 
                      let sk,sv,r' = spliceOutSuccessor r 
                      mk l sk sv r'
                else rebalance l k2 v2 (remove cf k r) 

        let rec mem (cf: IComparer<_>) k m = 
            match m with 
            | MapEmpty -> false
            | MapOne(k2,v2) -> (cf.Compare(k,k2) = 0)
            | MapNode(k2,v2,l,r,_) -> 
                let c = cf.Compare(k,k2) 
                if c < 0 then mem cf k l
                else (c = 0 || mem cf k r)

        let rec iter f m = 
            match m with 
            | MapEmpty -> ()
            | MapOne(k2,v2) -> f k2 v2
            | MapNode(k2,v2,l,r,_) -> iter f l; f k2 v2; iter f r

        let rec first f m = 
            match m with 
            | MapEmpty -> None
            | MapOne(k2,v2) -> f k2 v2 
            | MapNode(k2,v2,l,r,_) -> 
                match first f l with 
                | Some x as res -> res 
                | None -> 
                match f k2 v2 with 
                | Some x as res -> res 
                | None -> first f r

        let rec exists f m = 
            match m with 
            | MapEmpty -> false
            | MapOne(k2,v2) -> f k2 v2
            | MapNode(k2,v2,l,r,_) -> f k2 v2 || exists f l || exists f r

        let rec for_all f m = 
            match m with 
            | MapEmpty -> true
            | MapOne(k2,v2) -> f k2 v2
            | MapNode(k2,v2,l,r,_) -> f k2 v2 && for_all f l && for_all f r

        let rec map f m = 
            match m with 
            | MapEmpty -> empty
            | MapOne(k,v) -> MapOne(k,f v)
            | MapNode(k,v,l,r,h) -> let v2 = f v in MapNode(k,v2,map f l, map f r,h)

        let rec mapi f m = 
            match m with
            | MapEmpty -> empty
            | MapOne(k,v) -> MapOne(k,f k v)
            | MapNode(k,v,l,r,h) -> let v2 = f k v in MapNode(k,v2, mapi f l, mapi f r,h)

        let rec fold f m x = 
            match m with 
            | MapEmpty -> x
            | MapOne(k,v) -> f k v x
            | MapNode(k,v,l,r,h) -> fold f l (f k v (fold f r x))

        let foldSection (cf: IComparer<_>) lo hi f m x =
            let rec fold_from_to f m x = 
                match m with 
                | MapEmpty -> x
                | MapOne(k,v) ->
                    let clo_k = cf.Compare(lo,k)
                    let ck_hi = cf.Compare(k,hi)
                    let x = if clo_k <= 0 && ck_hi <= 0 then f k v x else x
                    x
                | MapNode(k,v,l,r,h) ->
                    let clo_k = cf.Compare(lo,k)
                    let ck_hi = cf.Compare(k,hi)
                    let x = if clo_k < 0                then fold_from_to f l x else x
                    let x = if clo_k <= 0 && ck_hi <= 0 then f k v x                     else x
                    let x = if ck_hi < 0                then fold_from_to f r x else x
                    x
           
            if cf.Compare(lo,hi) = 1 then x else fold_from_to f m x

        let rec fmap (cf: IComparer<_>) f m z acc = 
            match m with 
            | MapEmpty -> acc,z
            | MapOne(k,v) -> 
                let v',z = f k v z
                add cf k v' acc,z
            | MapNode(k,v,l,r,h) -> 
                let acc,z = fmap cf f r z acc
                let v',z = f k v z
                let acc = add cf k v' acc 
                fmap cf f l z acc

        let to_list m = fold (fun k v acc -> (k,v) :: acc) m []
        let to_array m = m |> to_list |> Array.of_list
        let of_list cf l = List.fold_left (fun acc (k,v) -> add cf k v acc) empty l

        let baked cf t =     { cf=(cf :> IComparer<'a>); tree=t }
        let fresh cf =       baked cf empty 
        let refresh s t =    { cf=s.cf;tree=t }
        let rec mkFromEnumerator cf acc (e : IEnumerator<_>) = 
            if e.MoveNext() then 
                let (x,y) = e.Current 
                mkFromEnumerator cf (add cf x y acc) e
            else acc
          
        let of_seq cf (c : #seq<_>) =
            mkFromEnumerator cf empty (c.GetEnumerator()) 
          
        let copyToArray s arr i =
            let j = ref i 
            s |> iter (fun x y -> arr.(!j) <- KeyValuePair(x,y); j := !j + 1)


        /// Imperative left-to-right iterators.
        type iterator<'key,'a> = 
             { /// invariant: always collapseLHS result 
               mutable stack: ('key,'a) MapTree list;  
               /// true when MoveNext has been called   
               mutable started : bool }

        // collapseLHS:
        // a) Always returns either [] or a list starting with SetOne.
        // b) The "fringe" of the set stack is unchanged. 
        let rec collapseLHS stack =
            match stack with
            | []                           -> []
            | MapEmpty             :: rest -> collapseLHS rest
            | MapOne (k,v)         :: rest -> stack
            | (MapNode(k,v,l,r,h)) :: rest -> collapseLHS (l :: MapOne (k,v) :: r :: rest)
          
        let mkIterator s = { stack = collapseLHS [s]; started = false }

        let not_started() = raise (new System.InvalidOperationException("Enumeration has not started. Call MoveNext."))
        let already_finished() = raise (new System.InvalidOperationException("Enumeration already finished."))

        let current i =
            if i.started then
                match i.stack with
                  | MapOne (k,v) :: _ -> new KeyValuePair<_,_>(k,v)
                  | []            -> already_finished()
                  | _             -> failwith "Please report error: Map iterator, unexpected stack for current"
            else
                not_started()

        let rec moveNext i =
          if i.started then
            match i.stack with
              | MapOne _ :: rest -> i.stack <- collapseLHS rest;
                                    i.stack <> []
              | [] -> false
              | _ -> failwith "Please report error: Map iterator, unexpected stack for moveNext"
          else
              i.started <- true;  (* The first call to MoveNext "starts" the enumeration. *)
              i.stack <> []

        let mkIEnumerator s = 
          let i = ref (mkIterator s) 
          { new IEnumerator<_> with 
                member self.Current = current !i
            interface System.Collections.IEnumerator with
                member self.Current = box (current !i)
                member self.MoveNext() = moveNext !i
                member self.Reset() = i :=  mkIterator s
            interface System.IDisposable with 
                member self.Dispose() = ()}


    open RawMapOps 

    module MapOps = 
        let baked cf t =     { cf=(cf :> IComparer<'a>); tree=t }
        let fresh cf =       baked cf empty 
        let refresh s t =    { cf=s.cf;tree=t }
    
    open MapOps 


    type Map<'key,'a> with 
        static member Empty() : Map<'key,'a> = fresh ComparisonIdentity.Structural
        member m.Add(k,v) : Map<'key,'a> = refresh m (add m.cf k v m.tree)
        member m.IsEmpty = is_empty m.tree
        member m.Find(k) = find m.cf k m.tree
        member m.Item with get(k : 'key) = find m.cf k m.tree
        member m.First(f) = first f m.tree 
        member m.Exists(f) = exists f m.tree 
        member m.Filter(f)  : Map<'key,'a> = filter m.cf f m.tree |> refresh m 
        member m.ForAll(f) = for_all f m.tree 
        member m.Fold f acc = fold f m.tree acc
        member m.FoldSection (lo:'key) (hi:'key) f (acc:'z) = foldSection m.cf lo hi f m.tree acc 
        member m.FoldAndMap f z  = let tree,z = fmap m.cf f m.tree z empty in refresh m tree,z
        member m.Iterate f = iter f m.tree
        member m.MapRange f  = refresh m (map f m.tree)
        member m.Map f  = refresh m (mapi f m.tree)
        member m.Partition(f)  : Map<'key,'a> * Map<'key,'a> = let r1,r2 = partition m.cf f m.tree  in refresh m r1, refresh m r2
        member m.Count = size m.tree
        member m.ContainsKey(k) = mem m.cf k m.tree
        member m.Remove(k)  : Map<'key,'a> = refresh m (remove m.cf k m.tree)
        member m.TryFind(k) = tryfind m.cf k m.tree
        member m.ToList() = to_list m.tree
        member m.ToArray() = to_array m.tree

    type Map<'key,'a> with 
        static member FromList(l) : Map<'key,'a> = 
           let cf = (ComparisonIdentity.Structural :> IComparer<'key>) 
           baked cf (of_list cf l)
        static member Create((ie : IEnumerable<_>)) : Map<'key,'a> = 
           let cf = (ComparisonIdentity.Structural :> IComparer<'key>) 
           baked cf (of_seq cf ie)
    
    type Map<'key,'a> with 
        static member Create() : Map<'key,'a> = Map.Empty()
        interface IEnumerable<KeyValuePair<'key, 'a>> with
            member s.GetEnumerator() = mkIEnumerator s.tree
        interface System.Collections.IEnumerable with
            override s.GetEnumerator() = (mkIEnumerator s.tree :> System.Collections.IEnumerator)
#if CLI_AT_MOST_1_1
#else
        interface IDictionary<'key, 'a> with 
            override s.Item 
                with get x = s.[x]            
                and  set x v = raise (NotSupportedException("Map values may not be mutated"))
            override s.Keys = ([| for kvp in s -> kvp.Key |] :> ICollection<'key>)
            override s.Values = ([| for kvp in s -> kvp.Value |] :> ICollection<'a>)
            override s.Add(k,v) = raise (NotSupportedException("Map values may not be mutated"))
            override s.ContainsKey(k) = s.ContainsKey(k)
            override s.TryGetValue(k,r) = if s.ContainsKey(k) then (r <- s.[k]; true) else false
            override s.Remove(k : 'key) = (raise (NotSupportedException("Map values may not be mutated")) : bool)

        interface ICollection<KeyValuePair<'key, 'a>> with 
            override s.Add(x) = raise (new System.NotSupportedException("Map values may not be mutated"));
            override s.Clear() = raise (new System.NotSupportedException("Map values may not be mutated"));
            override s.Remove(x) = raise (new System.NotSupportedException("Map values may not be mutated"));
            override s.Contains(x) = s.ContainsKey(x.Key) && s.[x.Key] = x.Value
            override s.CopyTo(arr,i) = copyToArray s.tree arr i
            override s.IsReadOnly = true
            override s.Count = Seq.length s
#endif
        interface System.IComparable with 
            member m1.CompareTo(m2: obj) = 
                Seq.compare 
                   (fun (kvp1 : KeyValuePair<_,_>) (kvp2 : KeyValuePair<_,_>)-> 
                       let c = m1.cf.Compare(kvp1.Key,kvp2.Key) in 
                       if c <> 0 then c else Operators.compare kvp1.Value kvp2.Value)
                   m1 (m2 :?> Map<'key,'a>)
        override this.Equals(that:obj) = ((this :> System.IComparable).CompareTo(that) = 0)
    

    [<System.Obsolete("Collections.IHashOps was an abbreviation for System.Collections.Generic.IEqualityComparer. Please use that name instead")>]
    type IHashOps<'key> = IEqualityComparer<'key> 


namespace Microsoft.FSharp.Collections.Tagged
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics
    open System

    #if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
    #else
    open System.Collections.Generic
    #endif

    type HashMultiMap<'key,'v,'hashTag>
         when 'hashTag :> IEqualityComparer<'key> 
         = { t : Microsoft.FSharp.Collections.HashMultiMap<'key,'v> } with 
        static member Create ((hasheq:'hashTag),stats,n) : HashMultiMap<'key,'v,'hashTag> = { t = HashTableOps.createTable stats hasheq n }
        static member Create ((hasheq:'hashTag),n) : HashMultiMap<'key,'v,'hashTag> = HashMultiMap.Create(hasheq ,HashStats.Create(),n)
        member x.Add(y,z) = HashTableOps.add x.t y z
        member x.Clear() = HashTableOps.clear x.t
        member x.Copy() :  HashMultiMap<'key,'v,'hashTag>  = { t = HashTableOps.copy x.t }
        member x.Find(y) = HashTableOps.find x.t y
        member x.Item with get(y : 'key) = HashTableOps.find x.t y
                      and  set (y:'key) (z:'v) = HashTableOps.replace x.t y z
        member x.FindAll(y) = HashTableOps.findAll x.t y
        member x.Fold f acc = HashTableOps.fold f x.t acc
        member x.Map  f :  HashMultiMap<'key,'v,'hashTag>  = { t  = HashTableOps.gen_map x.t f }
        member x.Iterate(f) =  HashTableOps.iter f x.t
        member x.Contains(y) = HashTableOps.mem x.t y
        member x.Remove(y) = HashTableOps.remove x.t y
        member x.Replace(y,z) = HashTableOps.replace x.t y z
        member x.TryFind(y) = HashTableOps.tryfind x.t y
        member x.Count = HashTableOps.count x.t 
        member x.GetLongestChain() =  HashTableOps.longestChain x.t.hmmBuckets 

    type HashSet<'a,'hashTag> 
         when 'hashTag :> IEqualityComparer<'a> =
    #if CLI_AT_MOST_1_1
        { t: HashMultiMap<'a,int,'hashTag> }
        static member Create(hasheq,n) : HashSet<'a,'hashTag> = {t = HashMultiMap.Create(hasheq,n) }
        static member Create((hasheq: #IEqualityComparer<'a>),stats,n) : HashSet<'a,'hashTag> = {t = HashMultiMap.Create(hasheq,stats,n) }
        member x.Add(y)    = x.t.Replace(y,0);
        member x.Clear() = x.t.Clear()
        member x.Copy() = {t = x.t.Copy() }
        member x.Fold f acc = x.t.Fold(fun y _ acc -> f y acc) acc
        member x.Map f = let t = HashMultiMap.Create(x.t.t.hmmOps,x.t.t.hmmStats,x.t.t.hmmCount) in t.Iterate(fun x _ -> t.Replace(f x,0)); { t = t }
        member x.Iterate(f) =  x.t.Iterate(fun x _ -> f x)
        member x.Contains(y) = x.t.Contains(y)
        member x.Remove(y) = x.t.Remove(y)
        member x.Count = x.t.Count
        member x.GetLongestChain() = List.map fst (x.t.GetLongestChain())
    #else
        { t: Dictionary<'a,int> }
        static member Create((hasheq: ('hashTag :> IEqualityComparer<'a>)),(n:int)) : HashSet<'a,'hashTag> = {t = new Dictionary<_,_>(n,hasheq) }
        static member Create((hasheq: ('hashTag :> IEqualityComparer<'a>)),(stats:HashStats),(n:int)) : HashSet<'a,'hashTag> = {t = new Dictionary<_,_>(n,hasheq) }
        member x.Add(y)    = x.t.[y] <- 0
        member x.Clear() = x.t.Clear()
        member x.Copy() : HashSet<'a,'hashTag>  = let t = new Dictionary<_,_>(x.t.Count,x.t.Comparer) in x.t |> Seq.iter (fun kvp -> t.[kvp.Key] <- 0); {t=t}
        member x.Fold f acc = x.t |> Seq.fold (fun acc kvp -> f kvp.Key acc) acc
        member x.Map f : HashSet<'a,'hashTag>  = let t = new Dictionary<_,_>(x.t.Count,x.t.Comparer) in x.t |> Seq.iter (fun kvp -> t.[f kvp.Key] <- 0); {t=t}
        member x.Iterate(f) =  x.t |> Seq.iter (fun kvp -> f kvp.Key)
        member x.Contains(y) = x.t.ContainsKey(y)
        member x.Remove(y) = x.t.Remove(y) |> ignore
        member x.Count = x.t.Count
        member x.GetLongestChain() = ([] : 'a list)
        interface IEnumerable<'a> with
          member x.GetEnumerator() = x.t.Keys.GetEnumerator() :> System.Collections.Generic.IEnumerator<_>
        interface System.Collections.IEnumerable with 
          member x.GetEnumerator() = x.t.Keys.GetEnumerator()  :> System.Collections.IEnumerator 
    #endif

    type Set<'a,'comparerTag> when 'comparerTag :> IComparer<'a> = 
        { cf: IComparer<'a>;
          tree: SetTree<'a> }
        interface System.IComparable 
        //interface IStructuralHash 
#if CLI_AT_MOST_1_1
#else
        interface ICollection<'a> 
#endif
        interface IEnumerable<'a> 
        interface System.Collections.IEnumerable 

    open RawSetOps
    module SetOps = 
        let baked cf t =     { cf=(cf :> IComparer<_>) ; tree=t }
        let refresh s t =    { cf=s.cf; tree=t }
        let fresh cf =       baked cf empty

    open SetOps

    type Set<'a,'comparerTag> with 
        // REVIEW: we can use .NET generics per-instantiation static fields to avoid allocating a new object for each empty
        // list (it will just be a quick lookup into a table of type-instantiation-indexed static fields).
        static member Empty(cf: 'comparerTag) : Set<'a,'comparerTag> = fresh cf 
        member s.Add(x) : Set<'a,'comparerTag> = refresh s (add s.cf x s.tree)
        member s.Remove(x) : Set<'a,'comparerTag> = refresh s (remove s.cf x s.tree)
        member s.Size = size s.tree
        member s.Count = size s.tree
        member s.Contains(x) = mem s.cf  x s.tree
        member s.Iterate(x) = iter  x s.tree
        member s.Fold f x  = fold f s.tree x
        member s.CheckBalanceInvariant = checkInvariant s.tree // diagnostics...
        member s.IsEmpty  = is_empty s.tree
        member s.Partition f  : Set<'a,'comparerTag> *  Set<'a,'comparerTag> = 
            match s.tree with 
            | SetEmpty -> s,s
            | _ -> let t1,t2 = partition s.cf f s.tree in refresh s t1, refresh s t2

        member s.Filter f  : Set<'a,'comparerTag> = 
            match s.tree with 
            | SetEmpty -> s
            | _ -> filter s.cf f s.tree |> refresh s

        member s.Exists f = exists f s.tree
        member s.ForAll f = for_all f s.tree
        static member (-) ((a: Set<'a,'comparerTag>),(b: Set<'a,'comparerTag>)) = Set.Difference(a,b)
        static member (+)  ((a: Set<'a,'comparerTag>),(b: Set<'a,'comparerTag>)) = Set.Union(a,b)
        static member Intersection((a: Set<'a,'comparerTag>),(b: Set<'a,'comparerTag>)) : Set<'a,'comparerTag>  = 
            match b.tree with 
            | SetEmpty -> b  (* A INTER 0 = 0 *)
            | _ -> 
               match a.tree with 
               | SetEmpty -> a (* 0 INTER B = 0 *)
               | _ -> intersection a.cf  a.tree b.tree |> refresh a
           
        static member Union((a: Set<'a,'comparerTag>),(b: Set<'a,'comparerTag>)) : Set<'a,'comparerTag>  = 
            match b.tree with 
            | SetEmpty -> a  (* A U 0 = A *)
            | _ -> 
               match a.tree with 
               | SetEmpty -> b  (* 0 U B = B *)
               | _ -> union a.cf  a.tree b.tree |> refresh a

        static member Difference((a: Set<'a,'comparerTag>),(b: Set<'a,'comparerTag>)) : Set<'a,'comparerTag>  = 
            match a.tree with 
            | SetEmpty -> a (* 0 - B = 0 *)
            | _ -> 
                match b.tree with 
                | SetEmpty -> a (* A - 0 = A *)
                | _ -> diff a.cf  a.tree b.tree |> refresh a

        static member Equality((a: Set<'a,'comparerTag>),(b: Set<'a,'comparerTag>)) = (RawSetOps.compare a.cf  a.tree b.tree = 0)
        static member Compare((a: Set<'a,'comparerTag>),(b: Set<'a,'comparerTag>)) = RawSetOps.compare a.cf  a.tree b.tree

        member x.Choose = choose x.tree
        member x.MinimumElement = minimumElement x.tree
        member x.MaximumElement = maximumElement x.tree

        member x.IsSubsetOf((y: Set<'a,'comparerTag>)) = subset x.cf x.tree y.tree 
        member x.IsSupersetOf((y: Set<'a,'comparerTag>)) = subset x.cf y.tree x.tree
        member x.ToList () = to_list x.tree
        member x.ToArray () = to_array x.tree


        interface System.IComparable with 
            member this.CompareTo(that: obj) = RawSetOps.compare this.cf this.tree ((that :?> Set<'a,'comparerTag>).tree)
        override this.Equals(that:obj) = ((this :> System.IComparable).CompareTo(that) = 0)
          
#if CLI_AT_LEAST_2_0
        interface ICollection<'a> with 
            member s.Add(x) = raise (new System.NotSupportedException("ReadOnlyCollection"))
            member s.Clear() = raise (new System.NotSupportedException("ReadOnlyCollection"))
            member s.Remove(x) = raise (new System.NotSupportedException("ReadOnlyCollection"))
            member s.Contains(x) = mem s.cf x s.tree
            member s.CopyTo(arr,i) = copyToArray s.tree arr i
            member s.get_IsReadOnly() = true
            member s.get_Count() = cardinal s.tree  
#endif
        interface IEnumerable<'a> with
            member s.GetEnumerator() = mkIEnumerator s.tree

        interface System.Collections.IEnumerable with
            override s.GetEnumerator() = (mkIEnumerator s.tree :> System.Collections.IEnumerator)

    type Set<'a,'comparerTag> with
        static member Singleton(cf,x) : Set<'a,'comparerTag>  = (Set.Empty(cf)).Add(x)
        static member Create((cf : 'comparerTag),(l : seq<'a>)) : Set<'a,'comparerTag> = 
          baked cf (of_seq (cf :> IComparer<'a>) l)


    open RawMapOps

#if CLI_AT_MOST_1_1
#else
    [<System.Diagnostics.DebuggerDisplay("#Entries = {Size}")  >]
#endif
    type Map<'key,'a,'comparerTag> when 'comparerTag :> IComparer<'key> =  
          { cf: IComparer<'key>; tree: MapTree<'key,'a> }
          interface System.IComparable
          interface IEnumerable<KeyValuePair<'key, 'a>>
          interface System.Collections.IEnumerable 

    module MapOps = 
        let baked cf t =     { cf=(cf :> IComparer<'a>); tree=t }
        let fresh cf =       baked cf empty 
        let refresh s t =    { cf=s.cf;tree=t }

    open MapOps 

    type Map<'key,'a,'comparerTag> with 
        static member Empty(cf : ('comparerTag :> IComparer<'key>)) : Map<'key,'a,'comparerTag> = fresh cf
        member m.Add(k,v) : Map<'key,'a,'comparerTag> = refresh m (add m.cf k v m.tree)
        member m.IsEmpty = is_empty m.tree
        member m.Find(k) = find m.cf k m.tree
        member m.Item with get(k : 'key) = find m.cf k m.tree
        member m.First(f) = first f m.tree 
        member m.Exists(f) = exists f m.tree 
        member m.Filter(f)  : Map<'key,'a,'comparerTag> = filter m.cf f m.tree |> refresh m 
        member m.ForAll(f) = for_all f m.tree 
        member m.Fold f acc = fold f m.tree acc
        member m.FoldSection (lo:'key) (hi:'key) f (acc:'z) = foldSection m.cf lo hi f m.tree acc 
        member m.FoldAndMap f z  : Map<'key,'b,'comparerTag> * _ = let tree,z = fmap m.cf f m.tree z empty in refresh m tree,z
        member m.Iterate f = iter f m.tree
        member m.MapRange f  : Map<'key,'b,'comparerTag> = refresh m (map f m.tree)
        member m.Map f  : Map<'key,'b,'comparerTag> = refresh m (mapi f m.tree)
        member m.Partition(f)  : Map<'key,'a,'comparerTag> * Map<'key,'a,'comparerTag> = let r1,r2 = partition m.cf f m.tree  in refresh m r1, refresh m r2
        member m.Count = size m.tree
        member m.ContainsKey(k) = mem m.cf k m.tree
        member m.Remove(k)  : Map<'key,'a,'comparerTag> = refresh m (remove m.cf k m.tree)
        member m.TryFind(k) = tryfind m.cf k m.tree
        member m.ToList() = to_list m.tree
        member m.ToArray() = to_array m.tree

    type Map<'key,'a,'comparerTag> with 
        static member FromList((cf : ('comparerTag :> IComparer<'key>)),l) : Map<'key,'a,'comparerTag> = baked cf (of_list (cf :> IComparer<_>) l)
        static member Create((cf : ('comparerTag :> IComparer<'key>)),(ie : IEnumerable<_>)) : Map<'key,'a,'comparerTag> = baked cf (of_seq (cf :> IComparer<_>) ie)
    
    type Map<'key,'a,'comparerTag> with 
        interface IEnumerable<KeyValuePair<'key, 'a>> with
            member s.GetEnumerator() = mkIEnumerator s.tree

        interface System.Collections.IEnumerable with
            override s.GetEnumerator() = (mkIEnumerator s.tree :> System.Collections.IEnumerator)

        interface System.IComparable with 
             member m1.CompareTo(m2: obj) = 
                 Seq.compare 
                   (fun (kvp1 : KeyValuePair<_,_>) (kvp2 : KeyValuePair<_,_>)-> 
                       let c = m1.cf.Compare(kvp1.Key,kvp2.Key) in 
                       if c <> 0 then c else Operators.compare kvp1.Value kvp2.Value)
                   m1 (m2 :?> Map<_,_,_>)
        override this.Equals(that:obj) = ((this :> System.IComparable).CompareTo(that) = 0)
    

namespace Microsoft.FSharp.Collections.Tagged
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Collections.Tagged
    open Microsoft.FSharp.Primitives.Basics
    open System
    #if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
    #else
    open System.Collections.Generic
    #endif

    type Map<'key,'a> = Map<'key, 'a, IComparer<'key>>    
    type Set<'a> = Set<'a, IComparer<'a>>    
    type HashSet<'a> = HashSet<'a, IEqualityComparer<'a>>    
    type HashMultiMap<'key,'a> = HashMultiMap<'key,'a, IEqualityComparer<'key>>    
    type MapUntagged<'key,'a> = Map<'key,'a>
    type SetUntagged<'a> = Set<'a>
    type HashSetUntagged<'a> = HashSet<'a>
    type HashMultiMapUntagged<'key,'a> = HashMultiMap<'key,'a>



namespace Microsoft.FSharp.Collections

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open System
    open Microsoft.FSharp.Primitives.Basics
#if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
#else
    open System.Collections.Generic
#endif


#if CLI_AT_MOST_1_1      
#else
    module CommonExtensions =  

        let (|KeyValue|) (kv: KeyValuePair<_,_>) = kv.Key, kv.Value
#endif
