// (c) Microsoft Corporation 2005-2007.  

#light

namespace Microsoft.FSharp.Math

open Microsoft.FSharp.Collections
open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Math
open Microsoft.FSharp.Math.Primitives
open System
open System.Globalization

type INumeric<'a> =
  interface
    abstract Zero: 'a
    abstract One: 'a
    abstract Add: 'a * 'a -> 'a
    abstract Subtract: 'a * 'a -> 'a
    abstract Multiply : 'a * 'a -> 'a
    abstract Negate : 'a -> 'a
    abstract Sign : 'a -> int
    abstract Abs : 'a -> 'a
    abstract ToString : 'a * string * System.IFormatProvider -> string
    abstract Parse : string * System.Globalization.NumberStyles * System.IFormatProvider -> 'a
  end

type IIntegral<'a> =
  interface
    inherit INumeric<'a>
    abstract Modulus: 'a * 'a -> 'a
    abstract Divide : 'a * 'a -> 'a
    abstract DivMod : 'a * 'a -> 'a * 'a
    //abstract Remainder: 'a * 'a -> 'a
    //abstract Quotient : 'a * 'a -> 'a
    //abstract QuotRem: 'a * 'a -> 'a * 'a
    abstract ToBigInt : 'a -> bigint
    abstract OfBigInt : bigint -> 'a
  end
  
type IFractional<'a> =
  interface
    inherit INumeric<'a>
    abstract Reciprocal : 'a -> 'a
    abstract Divide : 'a * 'a -> 'a
    //abstract OfBigNum : bignum -> 'a
  end

type IFloating<'a> =
  interface
    inherit IFractional<'a>
    abstract Pi : 'a
    abstract Exp : 'a -> 'a
    abstract Log : 'a -> 'a
    abstract Sqrt : 'a -> 'a
    abstract LogN : 'a * 'a -> 'a
    abstract Sin : 'a -> 'a
    abstract Cos : 'a -> 'a
    abstract Tan : 'a -> 'a
    abstract Asin : 'a -> 'a
    abstract Acos : 'a -> 'a
    abstract Atan : 'a -> 'a
    abstract Atan2 : 'a * 'a -> 'a
    abstract Sinh : 'a -> 'a
    abstract Cosh : 'a -> 'a
    abstract Tanh : 'a -> 'a

  end

type IIEEE<'a> =
  interface
    inherit IFloating<'a>
    abstract PositiveInfinity : 'a
    abstract NegativeInfinity : 'a
    abstract NaN              : 'a
    abstract Epsilon          : 'a

    abstract IsNaN: 'a -> bool 
    abstract IsInfinite : 'a -> bool 
    //abstract IsDenormalized   : 'a -> bool 
    //abstract IsNegativeZero   : 'a -> bool 
    //abstract IsIEEE           : 'a -> bool 
  end

type INormFloat<'a> =
  interface
    abstract Norm : 'a -> float
  end
 
module Instances = 
  let Int32Numerics = 
    { new IIntegral<int32> 
       with get_Zero() =0
       and  get_One() = 1
       and  Add(a,b) = a + b
       and  Subtract(a,b) = a - b
       and  Multiply(a,b) = a * b
       and  Negate(a) = - a 
       and  Abs(a) = a
       and  ToBigInt(a) = BigInt.of_int a
       and  OfBigInt(a) = BigInt.to_int a
       and  Sign(a) = Math.Sign(a)
       //and Remainder(a,b) = Int32.rem a b
       //and Quotient(a,b) = Int32.div a b
       and  Modulus(a,b) = a % b
       and  Divide(a,b) = a / b
       and  DivMod(a,b) = (a / b, a % b)
       and  ToString((x:int32),fmt,fmtprovider) = 
              x.ToString(fmt,fmtprovider) 
       and  Parse(s,numstyle,fmtprovider) = 
              System.Int32.Parse(s,numstyle,fmtprovider)
      interface INormFloat<int32>
       with  Norm(x) = float (abs x)
    }
  let Int64Numerics = 
    { new IIntegral<int64> 
       with get_Zero() =0L
       and  get_One() = 1L
       and  Add(a,b) = a + b
       and  Subtract(a,b) = a - b
       and  Multiply(a,b) = a * b
       and  Negate(a) = - a 
       and  Abs(a) = Math.Abs(a)
       and  ToBigInt(a) = BigInt.of_int64 a
       and  OfBigInt(a) = BigInt.to_int64 a
       and  Sign(a) = Math.Sign(a)
       //and Remainder(a,b) = Int32.rem a b
       //and Quotient(a,b) = Int32.div a b
       and  Modulus(a,b) = a % b
       and  Divide(a,b) = a / b
       and  DivMod(a,b) = (a / b, a % b)
       and  ToString((x:int64),fmt,fmtprovider) = x.ToString(fmt,fmtprovider) 
       and  Parse(s,numstyle,fmtprovider) = System.Int64.Parse(s,numstyle,fmtprovider)
      interface INormFloat<int64>
       with  Norm(x) = float (Math.Abs x)
    }
  let FloatNumerics = 
    { new IIEEE<float> 
       with get_Zero() = 0.0
       and  get_One() =  1.0
       and  Add(a,b) =  a + b
       and  Subtract(a,b) = a - b
       and  Multiply(a,b) = a * b
       and  get_PositiveInfinity() = Double.PositiveInfinity
       and  get_NegativeInfinity() = Double.NegativeInfinity
       and  get_NaN() = Double.NaN
       and  get_Epsilon() = Double.Epsilon
       and  IsInfinite(a) = Double.IsInfinity(a)
       and  IsNaN(a) = Double.IsNaN(a)
       and  get_Pi() = Math.PI
       and  Reciprocal(a) = 1.0/a
       and  Abs(a) = Math.Abs(a)
       and  Sign(a) = Math.Sign(a)
       and  Asin(a) = Math.Asin(a)
       and  Acos(a) = Math.Acos(a)
       and  Atan(a) = Math.Atan(a)
       and  Atan2(a,b) = Math.Atan2(a,b)
       and  Tanh(a) = Math.Tanh(a)
       and  Tan(a) = Math.Tan(a)
       and  Sqrt(a) = Math.Sqrt(a)
       and  Sinh(a) = Math.Sinh(a)
       and  Cosh(a) = Math.Cosh(a)
       and  Sin(a) = Math.Sin(a)
       and  Cos(a) = Math.Cos(a)
       and  LogN(a,n) = Math.Log(a,n)
       and  Log(a) = Math.Log(a)
       and  Exp(a) = Math.Exp(a)
       and  Negate(a) = -a 
       and  Divide(a,b) = a / b
       and  ToString((x:float),fmt,fmtprovider) = x.ToString(fmt,fmtprovider) 
       and  Parse(s,numstyle,fmtprovider) = System.Double.Parse(s,numstyle,fmtprovider)
      interface INormFloat<float>
       with  Norm(x) = float (Math.Abs x)
    }
  let Float32Numerics = 
    { new IFractional<float32> 
       with get_Zero() = 0.0f
       and  get_One() =  1.0f
       and  Add(a,b) = a + b
       and  Subtract(a,b) = a - b
       and  Multiply(a,b) = a * b
       and  Negate(a) = -a 
       and  Reciprocal(a) = 1.0f/a
       and  Sign(a) = Math.Sign(a)
       and  Abs(a) = Math.Abs(a)
       and  Divide(a,b) = a / b
       and  ToString((x:float32),fmt,fmtprovider) = x.ToString(fmt,fmtprovider) 
       and  Parse(s,numstyle,fmtprovider) = System.Single.Parse(s,numstyle,fmtprovider)
      interface INormFloat<float32>
       with  Norm(x) = float (Math.Abs x)
     }

  let BigNumNumerics = 
    { new IFractional<bignum> 
       with get_Zero() = BigNum.zero
       and  get_One() = BigNum.one
       and  Add(a,b)      = BigNum.add a b
       and  Subtract(a,b) = BigNum.sub a b
       and  Multiply(a,b) = BigNum.mul a b
       and  Divide(a,b)   = BigNum.div a b
       and  Abs(a) = BigNum.abs a
       and  Sign(a) = BigNum.sign a
       and  Negate(a) = BigNum.neg a 
       and  Reciprocal(a) = BigNum.div BigNum.one a 
                 // REVIEW: ignoring fmt, fmtprovider
       and  ToString((x:bignum),fmt,fmtprovider) = BigNum.to_string(x)
                 // REVIEW: ignoring numstyle, fmtprovider
       and  Parse(s,numstyle,fmtprovider) = BigNum.of_string(s)
      interface INormFloat<bignum>
       with  Norm(x) = BigNum.to_float (BigNum.abs x)
    }       

  let BigIntNumerics = 
    { new IIntegral<_> 
       with get_Zero() =BigInt.zero
       and  get_One() = BigInt.one
       and  Add(a,b) = BigInt.add a b
       and  Subtract(a,b) = BigInt.sub a b
       and  Multiply(a,b) = BigInt.mul a b
       and  Divide(a,b) = BigInt.div a b
       and  Negate(a) = BigInt.neg a 
       and  Modulus(a,b) = BigInt.rem a b
       and  DivMod(a,b) = BigInt.divmod a b
       and  Sign(a) = BigInt.sign a
       and  Abs(a) = BigInt.abs a
       and  ToBigInt(a) = a 
       and  OfBigInt(a) = a 
                 // REVIEW: ignoring fmt and fmtprovider
       and  ToString(x,fmt,fmtprovider) = BigInt.to_string(x) 
                 // REVIEW: ignoring fmt and fmtprovider
       and  Parse(s,numstyle,fmtprovider) = BigInt.of_string(s)
      interface INormFloat<bigint>
       with  Norm(x) = BigInt.to_float (BigInt.abs x)
    }       

namespace Microsoft.FSharp.Math.Types

    type INumeric<'a> = Microsoft.FSharp.Math.INumeric<'a>
    type IIntegral<'a> = Microsoft.FSharp.Math.IIntegral<'a>
    type IFractional<'a> = Microsoft.FSharp.Math.IFractional<'a>
    type IFloating<'a> = Microsoft.FSharp.Math.IFloating<'a>
    type IIEEE<'a> = Microsoft.FSharp.Math.IIEEE<'a>
