// (c) Microsoft Corporation 2005-2007. 

#light

namespace Microsoft.FSharp.Math

module GlobalAssociations 

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections
open Microsoft.FSharp.Math
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Math.Instances
open System

let ht = 
    let ht = new System.Collections.Hashtable() 
    let optab =
        [ (type float),   (Some(FloatNumerics    :> INumeric<float>) :> obj);
          (type int32),   (Some(Int32Numerics    :> INumeric<int32>) :> obj);
          (type int64),   (Some(Int64Numerics    :> INumeric<int64>) :> obj);
          (type bigint),  (Some(BigIntNumerics   :> INumeric<bigint>) :> obj);
          (type float32), (Some(Float32Numerics  :> INumeric<float32>) :> obj);
          (type Complex), (Some(Complex.Numerics :> INumeric<Complex>) :> obj);
          (type bignum),  (Some(BigNumNumerics   :> INumeric<bignum>) :> obj); ]
       
    List.iter (fun (ty,ops) -> ht.Add((ty :> obj),ops)) optab;
    ht

let Get(ty:System.Type) = ht.[box(ty)] 
    
let Put (ty: System.Type, d : obj)  =
    lock ht (fun () -> 
        if ht.Contains(box(ty)) then invalid_arg ("the type "+ty.Name+" already has a registered numeric association");
        ht.Add(box(ty), d))
  
let getNumericAssociation (h: ReifiedType<'a>) =
    let ty = h.result
    match Get ty  with 
    | null -> invalid_arg ("The type "+ty.Name+" has no associated dictionary of numeric operations")
    | :? (INumeric<'a> option) as r -> r
    | _ -> invalid_arg ("The type "+ty.Name+" has a dictionary but it was not of the right type")

let registerNumericAssociation (ty: ReifiedType<'a>) (d :> INumeric<'a>)  = Put(ty.result, box(Some d))

#if CLI_AT_MOST_1_1
let inline RegisterNumericAssociation (d : #INumeric< $a >)  = Put(typeof< ($a) >, box(Some d))
let inline GetNumericAssociationOption< $a >() = 
    let ty = typeof< ($a) > in 
    match Get(ty) with 
    | null -> None
    | :? (INumeric<'a> option) as r -> r
    | _ -> invalid_arg ("The type "+ty.Name+" has a dictionary but it was not of the right type")
let inline GetNumericAssociation() = (GetNumericAssociationOption()).Value
#else

let GetNumericAssociationOption<'a>() = 
    let ty = typeof<'a> in 
    match Get(ty) with 
    | null -> None
    | :? (INumeric<'a> option) as r -> r
    | _ -> invalid_arg ("The type "+ty.Name+" has a dictionary but it was not of the right type")
let GetNumericAssociation() = (GetNumericAssociationOption()).Value
let RegisterNumericAssociation (d :# INumeric<'a>)  = Put(typeof<'a>, box(Some d))
#endif


