(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
module Microsoft.Research.AbstractIL.Internal.Library 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
F#*)

let notlazy v = Lazy.lazy_from_val v

let isSome x = match x with None -> false | _ -> true
let isNone x = match x with None -> true | _ -> false
let isNil x = match x with [] -> true | _ -> false
let nonNil x = match x with [] -> false | _ -> true

(*IF-OCAML*)
let rec first f xs =
  match xs with 
  | [] -> None
  | h :: t -> 
     match f h with 
     | None -> first f t
     | Some x -> Some x


let option_bind f inp = match inp with None -> None | Some x -> f x
let rec choose_acc f xs acc =
  match xs with 
  | [] -> List.rev acc
  | h :: t -> 
     match f h with 
     | None -> choose_acc f t acc 
     | Some x -> choose_acc f t (x::acc)
let choose f xs = choose_acc f xs []

let (>>) f g x = g (f x)
let (<<) f g x = f(g(x))
let (|>) x f = f x

let rec list_mapi_aux f l n = 
  match l with [] -> [] | (h::t) -> f n h :: list_mapi_aux f t (n+1) 

let list_mapi f l = list_mapi_aux f l 0

let list_mapi2 f l1 l2 = 
  let rec go n x = match x with [],[] -> [] | (h1::t1),(h2::t2) -> f n h1 h2 :: go (n+1) (t1,t2) | _ -> failwith "list_mapi2" in 
  go 0 (l1,l2)

let rec list_iteri_aux f l n = 
  match l with [] -> () | (h::t) -> f n h; list_iteri_aux f t (n+1)
let list_iteri f l = list_iteri_aux f l 0

let rec array_exists_aux p l n =
  not (n >= Array.length l) && (p l.(n) || array_exists_aux p l (n+1))
  
let array_exists p l = array_exists_aux p l 0


(*ENDIF-OCAML*)


(*F#
let rec first f xs = List.first f xs
let option_bind f inp = Option.bind f inp
let choose f xs = List.choose f xs
let list_mapi f l = List.mapi f l
let list_mapi2 f l = List.mapi2 f l
let list_iteri f l = List.iteri f l 
let array_exists p l = Array.exists p l
F#*)

let list_removep f l = 
  let rec go x = 
    match x with 
      [] -> raise Not_found 
    | (h::t) -> if f h then h,t else let x,r = go t in x, h::r in 
  go l

let omap f opt = match opt with None -> None | Some x -> Some (f x)

let rec last l = match l with [] -> failwith "last" | [h] -> h | h::t -> last t

let rec range n m = 
  if n > m then [] else 
  n::range (n+1) m

let replicate x n = 
  Array.to_list (Array.make x n)

let chop n l = 
  let a = Array.of_list l in 
  Array.to_list (Array.sub a 0 n), 
  Array.to_list (Array.sub a n (Array.length a - n)) 
  
let insert e l =
  if List.mem e l then l else e::l

let rec remove e l =
  match l with
  | [] -> []
  | h::t -> if e = h then t else h :: remove e t

let diff l1 l2 =
  List.fold_right remove l2 l1

let union l1 l2 = 
  List.fold_right insert l1 l2

let rec index x = 
      let rec look n = function
            [] -> raise Not_found 
         | (h::t) -> if h=x then n else look (n+1) t in 
      look 0

let (???) = Int32.to_int 
let (!!!) = Int32.of_int 
let ( /./ ) = Int32.div 
let ( --- ) = Int32.sub 
let ( *** ) = Int32.mul 
let ( +++ ) = Int32.add 
let (|||) = Int32.logor 
let (^^^) = Int32.logxor
let (&&&) = Int32.logand 
let (<<<) = Int32.shift_left
let (lsr) = Int32.shift_right_logical

let rec front_n_back l = 
  match l with [] -> invalid_arg "front_n_back" 
  | [h] -> [],h
  | h::t -> let a,b = front_n_back t in h::a,b

let oiter f opt = match opt with None -> () | Some x -> f x

let ( >>>> ) x y = Int64.shift_right_logical x y 
let ( |||| ) x y = Int64.logor x y 
let ( &&&& ) x y = Int64.logand x y 
let ( !!!! ) x = Int64.of_int x 
let ( <<<< ) x y = Int64.shift_left x y 

(* get an initialization hole *)
let getH r = match !r with None -> failwith "getH" | Some x -> x

let rec rev_map_concat_onto f l acc = 
  match l with 
  | [] -> acc
  | h::t -> rev_map_concat_onto f t (List.rev_append (f h) acc)
  
let map_concat f l = List.rev (rev_map_concat_onto f l [])

let collecti f n = 
  let res = ref [] in 
  for i = 0 to n - 1 do 
    res := f i :: !res
  done; 
  List.rev !res

let rec array_forall_aux p l n =
  (n >= Array.length l) || (p l.(n) && array_forall_aux p l (n+1))
  
let array_forall p l = array_forall_aux p l 0

let rec array_exists_one_aux p l n =
  (n < Array.length l) && 
  (if p l.(n) then array_forall_aux (fun x -> not (p x)) l (n+1) else array_exists_one_aux p l (n+1))
  
let array_exists_one p l = array_exists_one_aux p l 0

(*---------------------------------------------------------------------------
!* Expanding mutable-lists-as-arrays
 *------------------------------------------------------------------------- *)
    

module ResizeArray = struct
(*IF-OCAML*)
  let the = function None -> failwith "the"  | Some x -> x

  type 'a t = 
    { mutable arr: 'a option array; 
      mutable curr: int }

  let length bb = bb.curr

  let create sz = 
    { arr= Array.create sz None; 
      curr = 0; }
  let ensure arrl new_size = 
    let old_arrl_size = Array.length arrl.arr in 
    if new_size > old_arrl_size then begin
      let old = arrl.arr in 
      arrl.arr <- Array.create (max new_size (old_arrl_size * 2)) None; 
      Array.blit old 0 arrl.arr 0 arrl.curr;
    end

  let to_array arrl = Array.map the (Array.sub arrl.arr 0 arrl.curr)

  let add arrl i = 
    let new_size = arrl.curr + 1 in 
    ensure arrl new_size;
    Array.set arrl.arr arrl.curr (Some i);
    arrl.curr <- new_size 

  let replace arrl i x = 
    if i < 0 || i >= arrl.curr then failwith "ResizeArray.replace";
    Array.set arrl.arr i (Some x)

  let get arrl i = 
    if i < 0 || i >= arrl.curr then failwith "ResizeArray.get";
    the (Array.get arrl.arr i)

(*ENDIF-OCAML*)

(*F#
#if CLI_AT_MOST_1_1
  let the = function None -> failwith "the"  | Some x -> x

  type 'a t = 
    { mutable arr: 'a option array; 
      mutable curr: int }

  let length bb = bb.curr

  let create sz = 
    { arr= Array.create sz None; 
      curr = 0; }
  let ensure arrl new_size = 
    let old_arrl_size = Array.length arrl.arr in 
    if new_size > old_arrl_size then begin
      let old = arrl.arr in 
      arrl.arr <- Array.create (max new_size (old_arrl_size * 2)) None; 
      Array.blit old 0 arrl.arr 0 arrl.curr;
    end

  let to_array arrl = Array.map the (Array.sub arrl.arr 0 arrl.curr)

  let add arrl i = 
    let new_size = arrl.curr + 1 in 
    ensure arrl new_size;
    Array.set arrl.arr arrl.curr (Some i);
    arrl.curr <- new_size 

  let replace arrl i x = 
    if i < 0 || i >= arrl.curr then failwith "ResizeArray.replace";
    Array.set arrl.arr i (Some x)

  let get arrl i = 
    if i < 0 || i >= arrl.curr then failwith "ResizeArray.get";
    the (Array.get arrl.arr i)

#else
// The F# implementation can use Array.zero_create to avoid options - the API ensures
// null values do not leak out.
  type 'a t = ResizeArray<'a>

  let length (bb:'a t) = bb.Count

  let create sz = new ResizeArray<'a>(sz:int)

  let to_array (arrl: 'a t) = Seq.to_array arrl
  
  let add (arrl:'a t) i = arrl.Add(i)

  let replace (arrl: 'a t) i x =  arrl.[i] <- x
  let get (arrl: 'a t) i =  arrl.[i] 
#endif
F#*)
  let choosei p arrl = 
    let rec aux i = 
      if i >= length arrl then None 
      else match p i (get arrl i) with 
      | None -> aux(i+1) 
      | res -> res in 
    aux 0
  let to_list arrl = Array.to_list (to_array arrl)

  let tryfind_index p arr = 
    let rec loop i = if i >= length arr then None else if p (get arr i) then Some(i) else loop(i+1) in
    loop 0

  
end
