//
// Many .NET executables come with unmanaged resources attached.
// (In particular nearly all the managed assemblies in the .NET
// Redistributable do so.)
//
// These are different to the "manifest resources" associated with
// .NET assemblies.      The unmanaged resources are bitmaps etc. 
// i.e. normal Win32 resources.
//
// Linking these into the executable is much the same
// as statically linking a .obj file into an executable.  The
// resources have no external links, but the resource data structure
// has a few internal RVA-based indirections.
//
// A complication is that the resource file can come in two flavours:
// semi-compiled and compiled.  The former is the ".res" format
// and the latter is the ".obj" format.  It's only really
// possible to do linking on the latter, so we call a utility
// to convert the .res format to the .obj format.  Luckily the
// Redist comes with such a utility...  
//
// At the moment we can only link in one resource file.  I don't know if
// that is a real limitation or not.  
//
// Boy is this gross.... Now you know that Abstract IL really does hide
// the grotty details from you....

#include "common.h"
#include <mscoree.h>
#include "cordefs.h"
#include "getcor.h"

// Get the Symbol entry given the head and a 0-based index
static IMAGE_SYMBOL* GetSymbolEntry(IMAGE_SYMBOL* pHead, int idx)
{
    return (IMAGE_SYMBOL*) (((BYTE*) pHead) + IMAGE_SIZEOF_SYMBOL * idx);
}

// LinkNativeResource: Given a resource blob "unlinkedResource" in .res or compiled
// .res (.obj) format write the bits we have to embed
// in a native image to the buffer "linkedResource", assuming the embedded bits will be 
// placed at "linkedResourceBaseRVA".  If "linkedResourceBaseRVA" is 0 then 
// just return the size, in order to allow the caller to allocate the buffer.
//
// 1. The linking portion of the code only links resources in obj format. 
// Thus if given a .res format then we must convert from .res to .obj 
// with CvtRes.exe.     CvtRes.exe is always available in the CLR installation directory.
//
// 2. Must touch up all COFF relocs from .rsrc$01 (resource header) to .rsrc$02
// (resource raw data)

#define MAKE_ANSIPTR_FROMWIDE(a,b) \
    char a[_MAX_PATH]; \
    WideCharToMultiByte(CP_ACP, 0, b, -1, a, _MAX_PATH, NULL, NULL);



EXTAPI resLinkNativeResource(BYTE *pbUnlinkedResource, 
                             DWORD nUnlinkedResourceSize, 
                             bool sscli, 
                             ULONG32 pbLinkedResourceBaseRVA, 
                             BYTE *pResBuffer, 
                             DWORD *pResBufferSize)
{
    // Ensure that resFileNameW is an object file (not just a binary resource)
    // If we convert, then return obj filename in szTempFileName

    if (*((DWORD*) pbUnlinkedResource) == 0x00000000)
    {
        //fprintf(stderr, "resLinkNativeResource: resource is a .res file\n"); fflush(stderr);
        //fprintf(stderr, "length(unlinkedResource = %d)\n", string_length(unlinkedResource)); fflush(stderr);

        WCHAR tempRes[MAX_PATH+1];
        WCHAR tempResPath[MAX_PATH+1];

        //fprintf(stderr, "resLinkNativeResource: nUnlinkedResourceSize = %d\n", nUnlinkedResourceSize); fflush(stderr);

        if (!GetTempPathW(MAX_PATH, tempResPath))
        {
            return E_FAIL;
            //failwith("GetTempPathW failed");
        }

        if (!GetTempFileNameW(tempResPath, L"RES", 0, tempRes))
        {
            return E_FAIL;
            //failwith("GetTempFileName failed");
        }

        WCHAR tempResObj[MAX_PATH+1];
        WCHAR tempResObjPath[MAX_PATH+1];

        if (!GetTempPathW(MAX_PATH, tempResObjPath))
        {
            return E_FAIL;
            //failwith("GetTempPathW failed");
        }

        if (!GetTempFileNameW(tempResObjPath, L"RES", 0, tempResObj))
        {
            return E_FAIL;
            //failwith("GetTempFileName failed");
        }


        HANDLE hResFile = CreateFileW(tempRes, GENERIC_WRITE,
            FILE_SHARE_WRITE, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

        // failure
        if (!hResFile || (hResFile == INVALID_HANDLE_VALUE))
        {
            return E_FAIL;
            //failwith("Can't create temporary resource file\n");
        }

        DWORD dwCount;

        BOOL fRet = WriteFile(hResFile,pbUnlinkedResource,nUnlinkedResourceSize,&dwCount,NULL);

        CloseHandle(hResFile);

        //fprintf(stderr, "Written .res file\n"); fflush(stderr);

        if (!fRet) 
        {
            return E_FAIL;
            //failwith("could not write resource file");
        }

        DWORD dwExitCode;

        PROCESS_INFORMATION pi;

        ULONG32 cchSystemDir = MAX_PATH + 1;
        WCHAR wszSystemDir[MAX_PATH + 1];
        IfFailRet(GetDllForEE(sscli), "getting DLL to discover CORSystemDirectory");

        GetCORSystemDirectoryProc *dllGetCorSysDir;
        if (!(dllGetCorSysDir = (GetCORSystemDirectoryProc *) GetProcAddress(g_RtDll, "GetCORSystemDirectory")))
        {
            return E_FAIL;
            //failwith("Couldn't find GetCORSystemDirectory");
        }

        IfFailRet(((*dllGetCorSysDir)(wszSystemDir,  _MAX_PATH,&cchSystemDir)), "calling CORSystemDirectory");

        STARTUPINFOA start;
        ZeroMemory(&start, sizeof(STARTUPINFOA));
        start.cb = sizeof(STARTUPINFOA);
        start.dwFlags = STARTF_USESHOWWINDOW;
        start.wShowWindow = SW_HIDE;
        char* szMachine;
        /*
        if(pewriter.isIA64())
        szMachine = "IA64";
        else if(pewriter.isAMD64())
        SzMachine = "AMD64";
        else
        */
        szMachine = "IX86";

        // Res file, so convert it
        char szCmdLine[_MAX_PATH<<1];

        MAKE_ANSIPTR_FROMWIDE(pSystemDir, wszSystemDir);
        MAKE_ANSIPTR_FROMWIDE(pTempObjFile, tempResObj);
        MAKE_ANSIPTR_FROMWIDE(pTempResFile, tempRes);

        sprintf(szCmdLine,
            "%scvtres.exe /NOLOGO /READONLY /MACHINE:%s \"/OUT:%s\" \"%s\"",
            pSystemDir,
            szMachine,
            pTempObjFile,
            pTempResFile);

        //        fprintf(stderr, "Command line: %s\n", szCmdLine); fflush(stderr);

        if (!CreateProcessA(
            NULL,
            szCmdLine,
            NULL,
            NULL,
            true,
            0,
            0,
            NULL,
            &start,
            &pi))
        {
            return E_FAIL;
            //failwith("error: could not run cvtres.exe");
        }

        // If process runs, wait for it to finish
        WaitForSingleObject(pi.hProcess, INFINITE);

        GetExitCodeProcess(pi.hProcess, &dwExitCode);

        CloseHandle(pi.hProcess);

        CloseHandle(pi.hThread);

        if (dwExitCode != 0)
        {       // CvtRes.exe ran, but failed
            return E_FAIL;
            //failwith("cvtres.exe failed");
        }

        // Conversion succesful, so now read .obj file.

        HANDLE hObjFile = CreateFileW(tempResObj, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
        if (hObjFile == INVALID_HANDLE_VALUE) 
        {
            return E_FAIL;
            //failwith ("Resource file not found\n");
        }

        DWORD unlinkedObjSize = GetFileSize(hObjFile, NULL);
        pbUnlinkedResource = (BYTE *) malloc(sizeof(BYTE *) * unlinkedObjSize);

        fRet = ReadFile(hObjFile,pbUnlinkedResource,unlinkedObjSize,&dwCount,NULL);

        CloseHandle(hObjFile);

        if (!fRet) 
        {
            return E_FAIL;
            //failwith("could not read resource file");
        }

        //fprintf(stderr, "Done reading .obj resource file, length(unlinkedResource = %d)\n", string_length(unlinkedResource)); fflush(stderr);

        // delete temporary file if we created one
        DeleteFileW(tempResObj);
        DeleteFileW(tempRes);


    }

    // read the resource file and spit it out in the .rsrc section

    IMAGE_FILE_HEADER *hMod = (IMAGE_FILE_HEADER*)pbUnlinkedResource;

    if (hMod->SizeOfOptionalHeader != 0) 
    {
            return E_FAIL;
        //failwith ("Invalid .res file: Illegal optional header");
    }

    // first section is directly after header
    IMAGE_SECTION_HEADER *pSection = (IMAGE_SECTION_HEADER *)(hMod+1);
    IMAGE_SECTION_HEADER *rsrc01 = NULL;        // resource header
    IMAGE_SECTION_HEADER *rsrc02 = NULL;        // resource data
    for (int i=0; i < hMod->NumberOfSections; i++) 
    {
        if (strcmp(".rsrc$01", (char *)(pSection+i)->Name) == 0) 
        {
            rsrc01 = pSection+i;
        } 
        else if (strcmp(".rsrc$02", (char *)(pSection+i)->Name) == 0) 
        {
            rsrc02 = pSection+i;
        }
    }
    if (!rsrc01 || !rsrc02)
    {
            return E_FAIL;
        //failwith (" Invalid .res file: Missing sections .rsrc$01 or .rsrc$02\n");
    }

    int size = rsrc01->SizeOfRawData + rsrc02->SizeOfRawData;

    //fprintf(stderr, "Size of linked resource will be %d, rsrc01->SizeOfRawData = %d, rsrc02->SizeOfRawData = %d\n", size, rsrc01->SizeOfRawData, rsrc02->SizeOfRawData);

    if (pResBufferSize)
        *pResBufferSize = size;

    if (pbLinkedResourceBaseRVA == 0 || !pResBuffer)
        return S_OK;


    // Copy resource header
    memcpy(pResBuffer, (char *)hMod + rsrc01->PointerToRawData, rsrc01->SizeOfRawData);

    // map all the relocs in .rsrc$01 using the reloc and symbol tables in the COFF object.,

    const int nTotalRelocs = rsrc01->NumberOfRelocations;
    const IMAGE_RELOCATION* pReloc = (IMAGE_RELOCATION*) ((BYTE*) hMod + (rsrc01->PointerToRelocations));
    IMAGE_SYMBOL* pSymbolTableHead = (IMAGE_SYMBOL*) (((BYTE*)hMod) + hMod->PointerToSymbolTable);

    DWORD dwOffsetInRsrc2;
    for(int iReloc = 0; iReloc < nTotalRelocs; iReloc ++, pReloc++)
    {
        // Compute Address where RVA is in $01
        DWORD* pAddress = (DWORD*) (((BYTE*) hMod) + rsrc01->PointerToRawData + pReloc->VirtualAddress);

        // index into symbol table, provides address into $02
        DWORD IdxSymbol = pReloc->SymbolTableIndex;
        IMAGE_SYMBOL* pSymbolEntry = GetSymbolEntry(pSymbolTableHead, IdxSymbol);

        // Ensure the symbol entry is valid for a resource.
        if ((pSymbolEntry->StorageClass != IMAGE_SYM_CLASS_STATIC) ||
            (pSymbolEntry->Type != IMAGE_SYM_TYPE_NULL) ||
            (pSymbolEntry->SectionNumber != 3)) // 3rd section is .rsrc$02
        {
            return E_FAIL;
            //failwith("Invalid .res file: Illegal symbol entry\n");
        }

        // Ensure that RVA is valid address (inside rsrc02)
        if (pSymbolEntry->Value >= rsrc02->SizeOfRawData)
        {
            return E_FAIL;
            //failwith("Invalid .res file: Illegal rva into .rsrc$02\n");
        }
        *(DWORD*)(pResBuffer + pReloc->VirtualAddress) = 
            pbLinkedResourceBaseRVA+rsrc01->SizeOfRawData+pSymbolEntry->Value;
    }

    // Copy $02 (resource raw) into the pResBuffer
    memcpy(pResBuffer+rsrc01->SizeOfRawData, (char *)hMod + rsrc02->PointerToRawData, rsrc02->SizeOfRawData);

    //fprintf(stderr, "Done writing linked resource...\n", size);

    return S_OK;

}


/* --------------------------------------------------------------------------------
* resUnlinkNativeResource -  Reading a native resource from a .NET binary.
*
* Unlink a resource embedded in an image to produce a buffer
* corresponding to a .res file.
* 
* linkedResource should be a buffer containing the contents of the
* resource directory.  It is in a section that has, at least, the 
* bits 'IMAGE_SCN_CNT_INITIALIZED_DATA' and 'IMAGE_SCN_MEM_READ' set.
*
* The structure of the resource directory consists of 
* a root directory (defined by the structure IMAGE_RESOURCE_DIRECTORY immediately 
* followed by a number of resource directory entries (defined by the structure 
* IMAGE_RESOURCE_DIRECTORY_ENTRY). These are defined thus:
* 
* Private Type IMAGE_RESOURCE_DIRECTORY
*    Characteristics As Long '\\Seems to be always zero?
*    TimeDateStamp As Long
*    MajorVersion As Integer
*    MinorVersion As Integer
*    NumberOfNamedEntries As Integer
*    NumberOfIdEntries As Integer
* End Type
* 
* Private Type IMAGE_RESOURCE_DIRECTORY_ENTRY
*     dwName As Long
*     dwDataOffset As Long
*     CodePage As Long
*     Reserved As Long
* End Type
*
* Each resource directory entry can either point to the actual resource data 
* or to another layer of resource directory entries. If the highest bit of
* dwDataOffset is set then this points to a directory otherwise it points to 
* the resource data.
*
* Documentation on the .res format (ResFormat) which we write is harder to find...
* -------------------------------------------------------------------------------- */

struct ResFormatHeader
{
    DWORD       dwDataSize;
    DWORD       dwHeaderSize;
    DWORD       dwTypeID;
    DWORD       dwNameID;
    DWORD       dwDataVersion;
    WORD        wMemFlags;
    WORD        wLangID;
    DWORD       dwVersion;
    DWORD       dwCharacteristics;
    ResFormatHeader()
    {
        memset(this,0,sizeof(ResFormatHeader));
        dwHeaderSize = sizeof(ResFormatHeader);
        dwTypeID = dwNameID = 0xFFFF;
    };
};

struct ResFormatNode
{
    ResFormatHeader     ResHdr;
    IMAGE_RESOURCE_DATA_ENTRY DataEntry;
    unsigned cType;
    WCHAR* wzType;
    unsigned cName;
    WCHAR* wzName;
    ResFormatNode(DWORD tid, DWORD nid, DWORD lid, DWORD dataOffset, BYTE* pbLinkedResource)
    {
        if(tid & 0x80000000)
        {
            ResHdr.dwTypeID = 0;
            tid &= 0x7FFFFFFF;
            cType = *((WORD*)(pbLinkedResource+tid));
            
            wzType = new WCHAR[cType+1];
            memcpy(wzType,pbLinkedResource+tid+sizeof(WORD),cType*sizeof(WCHAR));
            wzType[cType]=0;
        }
        else
        {
            ResHdr.dwTypeID = (0xFFFF |((tid & 0xFFFF)<<16));
            wzType = NULL;
            cType = 0;
        }
        
        if(nid & 0x80000000)
        {
            ResHdr.dwNameID = 0;
            nid &= 0x7FFFFFFF;
            cName = *((WORD*)(pbLinkedResource+nid));
            wzName = new WCHAR[cName+1];
            memcpy(wzName, pbLinkedResource+nid+sizeof(WORD), cName*sizeof(WCHAR));
            wzName[cName]=0;
        }
        else
        {
            ResHdr.dwNameID = (0xFFFF |((nid & 0xFFFF)<<16));
            wzName = NULL;
            cName = 0;
        }
        
        ResHdr.wLangID = (WORD)lid;
        memcpy(&DataEntry,(pbLinkedResource+dataOffset),sizeof(IMAGE_RESOURCE_DATA_ENTRY));
        ResHdr.dwDataSize = DataEntry.Size;
    };
    int Save(ULONG32 pbLinkedResourceBaseRVA, BYTE *pbLinkedResource, BYTE *pUnlinkedResource)
    {
        // Dump them to pUnlinkedResource
        // For each resource write header and data
        int size = 0;
        ResHdr.dwHeaderSize = sizeof(ResFormatHeader);
        if(wzType) 
        {
            ResHdr.dwHeaderSize += (cType + 1)*sizeof(WCHAR) - sizeof(DWORD);
        }
        if(wzName) 
        {
            ResHdr.dwHeaderSize += (cName + 1)*sizeof(WCHAR) - sizeof(DWORD);
        }
#define SaveChunk(p,sz) \
        { if (pUnlinkedResource) {\
        memcpy(pUnlinkedResource, p, sz); \
        pUnlinkedResource += sz; } \
        size += sz; }

        //---- Constant part of the header: DWORD,DWORD
        SaveChunk(&ResHdr.dwDataSize, sizeof(DWORD));
        SaveChunk(&ResHdr.dwHeaderSize, sizeof(DWORD));

        //--- Variable part of header: type and name
    DWORD dwFiller = 0;
        if(wzType)
        {
            SaveChunk(wzType,(cType + 1)*sizeof(WCHAR));
        dwFiller += cType + 1;
        }
        else
        {
            SaveChunk(&ResHdr.dwTypeID,sizeof(DWORD));
        }
        if(wzName)
        {
            SaveChunk(wzName,(cName + 1)*sizeof(WCHAR));
        dwFiller += cName + 1;
        }
        else
        {
            SaveChunk(&ResHdr.dwNameID,sizeof(DWORD));
        }

    BYTE          bNil[3] = {0,0,0};

    // Align remaining fields on DWORD (nb. poor bit twiddling code taken from ildasm's dres.cpp)
    if (dwFiller & 1)
            SaveChunk(bNil,2);
  
        //---- Constant part of the header: DWORD,WORD,WORD,DWORD,DWORD
        SaveChunk(&ResHdr.dwDataVersion,8*sizeof(WORD));

        //---- Header done, now data
        BYTE* pbData = pbLinkedResource + DataEntry.OffsetToData - pbLinkedResourceBaseRVA;

        //fprintf(stderr, "resUnlinkNativeResource: DataEntry.Size = %d\n", DataEntry.Size); fflush(stderr);

        SaveChunk(pbData,DataEntry.Size);
    dwFiller = DataEntry.Size & 3;
        if(dwFiller)
        {
            dwFiller = 4 - dwFiller;
            BYTE          bNil[3] = {0,0,0};
            SaveChunk(bNil,dwFiller);
        }
    
        return size;
    };
};



EXTAPI resUnlinkNativeResource(ULONG32 pbLinkedResourceBaseRVA, 
                               BYTE *pbLinkedResource, 
                               DWORD linkedResourceSize, 
                               BYTE *pResBuffer, 
                               DWORD *pResBufferSize)
{

    unsigned ulNumResNodes=0;
    // First, pull out all linkedResource nodes (tree leaves), see ResFormatNode struct

        //fprintf(stderr, "resUnlinkNativeResource: linkedResourceSize = %d\n", linkedResourceSize); fflush(stderr);

    PIMAGE_RESOURCE_DIRECTORY pirdType = (PIMAGE_RESOURCE_DIRECTORY)pbLinkedResource;
    PIMAGE_RESOURCE_DIRECTORY_ENTRY pirdeType = (PIMAGE_RESOURCE_DIRECTORY_ENTRY)(pbLinkedResource+sizeof(IMAGE_RESOURCE_DIRECTORY));
    unsigned N = pirdType->NumberOfNamedEntries+pirdType->NumberOfIdEntries;

    unsigned int nResNodes = 0;

    for(unsigned int i=0; i < N; i++)
    {
        if(pirdeType[i].DataIsDirectory)
        {
            BYTE*       pbNameBase = pbLinkedResource + pirdeType[i].OffsetToDirectory;
            PIMAGE_RESOURCE_DIRECTORY pirdName = (PIMAGE_RESOURCE_DIRECTORY)pbNameBase;
            PIMAGE_RESOURCE_DIRECTORY_ENTRY pirdeName = (PIMAGE_RESOURCE_DIRECTORY_ENTRY)(pbNameBase+sizeof(IMAGE_RESOURCE_DIRECTORY));
            unsigned N2 = pirdName->NumberOfNamedEntries+pirdName->NumberOfIdEntries;
            
            for(unsigned j=0; j < N2; j++)
            {
                if(pirdeName[j].DataIsDirectory)
                {
                    BYTE*       pbLangBase = pbLinkedResource + pirdeName[j].OffsetToDirectory;
                    PIMAGE_RESOURCE_DIRECTORY pirdLang = (PIMAGE_RESOURCE_DIRECTORY)pbLangBase;
                    unsigned short N3 = pirdLang->NumberOfNamedEntries+pirdLang->NumberOfIdEntries;
                                        
                    nResNodes += N3;
                }
                else
                {
                    nResNodes++;
                }
            }
        }
        else
        {
            nResNodes++;
        }
    }

        //fprintf(stderr, "resUnlinkNativeResource: nResNodes = %d\n", nResNodes); fflush(stderr);

    ResFormatNode* pResNodes = (ResFormatNode *) alloca(nResNodes * sizeof(ResFormatNode));
    memset(pResNodes,0,nResNodes * sizeof(ResFormatNode));

    for(unsigned int i=0; i < N; i++)
    {
        DWORD dwTypeID = pirdeType[i].Name;
        if(pirdeType[i].DataIsDirectory)
        {
            BYTE*       pbNameBase = pbLinkedResource + pirdeType[i].OffsetToDirectory;
            PIMAGE_RESOURCE_DIRECTORY pirdName = (PIMAGE_RESOURCE_DIRECTORY)pbNameBase;
            PIMAGE_RESOURCE_DIRECTORY_ENTRY pirdeName = (PIMAGE_RESOURCE_DIRECTORY_ENTRY)(pbNameBase+sizeof(IMAGE_RESOURCE_DIRECTORY));
            unsigned N2 = pirdName->NumberOfNamedEntries+pirdName->NumberOfIdEntries;
            
            for(unsigned j=0; j < N2; j++)
            {
                DWORD dwNameID = pirdeName[j].Name;
                if(pirdeName[j].DataIsDirectory)
                {
                    BYTE*       pbLangBase = pbLinkedResource + pirdeName[j].OffsetToDirectory;
                    PIMAGE_RESOURCE_DIRECTORY pirdLang = (PIMAGE_RESOURCE_DIRECTORY)pbLangBase;
                    PIMAGE_RESOURCE_DIRECTORY_ENTRY pirdeLang = (PIMAGE_RESOURCE_DIRECTORY_ENTRY)(pbLangBase+sizeof(IMAGE_RESOURCE_DIRECTORY));
                    unsigned N3 = pirdLang->NumberOfNamedEntries+pirdLang->NumberOfIdEntries;
                    
                    for(unsigned k=0; k < N3; k++)
                    {
                        DWORD dwLangID = pirdeLang->Name;
                        if(pirdeLang[k].DataIsDirectory)
                        {
                            return E_FAIL;
                            //failwith("Resource hierarchy exceeds three levels");
                        }
                        else
                        {
                            pResNodes[ulNumResNodes++] = ResFormatNode(dwTypeID,dwNameID,dwLangID, pirdeLang[k].OffsetToData,pbLinkedResource);
                        }
                    }
                }
                else
                {
                    pResNodes[ulNumResNodes++] = ResFormatNode(dwTypeID,dwNameID,0,pirdeName[j].OffsetToData,pbLinkedResource);
                }
            }
        }
        else
        {
            pResNodes[ulNumResNodes++] = ResFormatNode(dwTypeID,0,0,pirdeType[i].OffsetToData,pbLinkedResource);
        }
    }
    
    // OK, all tree leaves are in ResFormatNode structs, and ulNumResNodes ptrs are in pResNodes
    unsigned int size = 0;
    if(ulNumResNodes)
    {
        size += sizeof(ResFormatHeader);
        for(unsigned int i=0; i < ulNumResNodes; i++)
        {
            size += pResNodes[i].Save(pbLinkedResourceBaseRVA, pbLinkedResource, NULL);
        }
    }
    
    if (pResBufferSize) 
        *pResBufferSize = size;
    
        //fprintf(stderr, "resUnlinkNativeResource: size = %d\n", size); fflush(stderr);

    if (pResBuffer) 
    {
        BYTE *pResBufferPtr = pResBuffer;
        
        if(ulNumResNodes)
        {
            
            // Write dummy header
            ResFormatHeader rh = ResFormatHeader();
            memcpy(pResBufferPtr,&rh,sizeof(ResFormatHeader));
            pResBufferPtr += sizeof(ResFormatHeader);
            // For each linkedResource write header and data
            for(unsigned int i=0; i < ulNumResNodes; i++)
            {
                pResBufferPtr += pResNodes[i].Save(pbLinkedResourceBaseRVA, pbLinkedResource, pResBufferPtr);
            }
            if ((pResBufferPtr - pResBuffer) != size)
            {
                return E_FAIL;
                //failwith("Sizes are not correct");
            }
        }
    }
    return S_OK;

}


#ifdef CAML_STUBS

//-----------------------------------------------------------------------------
// OCaml stubs for the above
//-----------------------------------------------------------------------------


extern "C"
CAMLprim value resLinkNativeResourceCaml(value unlinkedResource, value sscli, value linkedResourceBaseRVA, value linkedResource)
{
    CAMLparam4(unlinkedResource, sscli, linkedResourceBaseRVA, linkedResource);

    //fprintf(stderr, "LinkResourceFile\n"); fflush(stderr);

    ULONG32 pbLinkedResourceBaseRVA = Int32_val(linkedResourceBaseRVA);
    DWORD nUnlinkedResourceSize = string_length(unlinkedResource);
    BYTE *pbUnlinkedResource = (BYTE *) String_val(unlinkedResource);
    BYTE *pbLinkedResourceBuffer = (BYTE *) String_val(linkedResource);

    DWORD cSize = 0;
    IfFailThrow(resLinkNativeResource(pbUnlinkedResource, nUnlinkedResourceSize,Bool_val(sscli),pbLinkedResourceBaseRVA,pbLinkedResourceBuffer,&cSize),"linking native resource");
    CAMLreturn(copy_int32(cSize));

}



extern "C"
CAMLprim value resUnlinkNativeResourceCaml(value linkedResourceBaseRVA, value linkedResource)
{

    CAMLparam2(linkedResourceBaseRVA, linkedResource);

    ULONG32 pbLinkedResourceBaseRVA = Int32_val(linkedResourceBaseRVA);
    DWORD nLinkedResourceSize = string_length(linkedResource);
    BYTE *pbLinkedResource = (BYTE *) String_val(linkedResource);
    DWORD cSize = 0;
    IfFailThrow(resUnlinkNativeResource(pbLinkedResourceBaseRVA,pbLinkedResource,nLinkedResourceSize,NULL,&cSize),"getting size of unlinked resource");
    CAMLlocal1(res);
    res = alloc_string(cSize);
    BYTE *pResBuffer = (BYTE *) String_val(res);
    IfFailThrow(resUnlinkNativeResource(pbLinkedResourceBaseRVA,pbLinkedResource,nLinkedResourceSize,pResBuffer,&cSize),"getting unlinked resource");
    CAMLreturn(res);
}

#endif
