(* (c) Microsoft Corporation. All rights reserved  *)

(*F# 
module Microsoft.Research.AbstractIL.Internal.Zset 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
F#*)

type 'a order = 'a -> 'a -> int

(*F#
type set<'a> = Microsoft.FSharp.Collections.Tagged.Set<'a >
type 'a t = set<'a>

let empty (ord : 'a order) = Microsoft.FSharp.Collections.Tagged.Set.Empty(ComparisonIdentity.Custom ord)

let is_empty (s:set<_>) = s.IsEmpty

let mem x (s:set<_>) = s.Contains(x)
let add x (s:set<_>) = s.Add(x)
let addL xs a = List.fold_left (fun a x -> add x a) a xs
    
let singleton ord x = add x (empty ord)
let remove x (s:set<_>) = s.Remove(x)

let fold (f : 'a -> 'b -> 'b) (s:set<_>) b = s.Fold f b
let iter f (s:set<_>) = s.Iterate f 
let for_all p (s:set<_>) = s.ForAll p 
let forall  p (s:set<_>) = s.ForAll p 
let exists  p (s:set<_>) = s.Exists p 
let subset (s1:set<_>) (s2:set<_>)  = s1.IsSubsetOf s2
let equal (s1:set<_>) (s2:set<_>)  = Microsoft.FSharp.Collections.Tagged.Set.Equality(s1,s2)
let elements (s:set<_>) = s.ToList()
let filter p (s:set<_>) = s.Filter p

let union (s1:set<_>) (s2:set<_>)  = Microsoft.FSharp.Collections.Tagged.Set.Union(s1,s2)
let inter (s1:set<_>) (s2:set<_>)  = Microsoft.FSharp.Collections.Tagged.Set.Intersection(s1,s2)
let diff (s1:set<_>) (s2:set<_>)  = Microsoft.FSharp.Collections.Tagged.Set.Difference(s1,s2)
F#*)

(*IF-OCAML*)
type 'a set = Zset of ('a,unit) Zmap.map
type 'a t = 'a set

let empty ord = Zset (Zmap.empty ord)
let is_empty (Zset a) = Zmap.is_empty a

let mem x (Zset y) = Zmap.mem x y
let add x (Zset y) = Zset (Zmap.add x () y)
let addL xs a = List.fold_left (fun a x -> add x a) a xs
    
let singleton ord x = add x (empty ord)
let remove x (Zset y) = Zset (Zmap.remove x y)

let fold (f : 'a -> 'b -> 'b) (Zset a) b = Zmap.fold (fun x _ y -> f x y) a b
let iter f (Zset a) = Zmap.iter (fun x _ -> f x) a
let for_all p (Zset a) = Zmap.forall (fun x () -> p x) a
let forall  p (Zset a) = Zmap.forall (fun x () -> p x) a
let exists  p (Zset a) = Zmap.exists (fun x () -> p x) a
let subset a b  = for_all (fun x -> mem x b) a
let equal a b  = subset a b & subset b a
let elements (Zset a) = Zmap.keys a
let filter p s = fold (fun x ss -> if p x then ss else remove x ss) s s

let union (Zset a) b = Zmap.fold (fun x  _ y -> add x y) a b
let inter a b =
 (* fold over xa:a removing xa if it is not in b *)
  fold (fun xa acc ->
        if mem xa b then acc
	            else remove xa acc) a a
let diff a b =
  (* fold over xb:b removing xb from a (only if member) *)
  fold (fun xb a -> if mem xb a then remove xb a else a) b a
(*ENDIF-OCAML*)

(*    
let a = List.fold_right add [1;2;3;4;5;6;7;8;9;10] (empty Pervasives.compare)
let b = List.fold_right add [6;7;8;9;10;11;12;13;14;15] (empty Pervasives.compare)
let xs = elements a
let xs = elements (inter a b)
let xs = elements (diff a b)
let xs = elements (diff b a)
*)

let mem_of m k = mem k m
