(* (c) Microsoft Corporation. All rights reserved *)


(*-------------------------------------------------------------------------
!* Incremental type inference constraint solving.  
 *
 * Primary constraints are:
 *   - type equations        ty1 = ty2
 *   - subtype inequations   ty1 :> ty2
 *   - trait constraints     tyname : (static member op_Addition : 'a * 'b -> 'c)
 *
 * Plus some other constraints inherited from .NET generics.
 * 
 * The constraints are immediately processed into a normal form, in particular
 *   - type equations on inference parameters:   'tp = ty
 *   - type inequations on inference parameters: 'tp :> ty
 *   - other constraints on inference paramaters
 *
 * The state of the inference engine is kept in imperative mutations to inference
 * type variables.
 *
 * The use of the normal form allows the state of the inference engine to 
 * be queried for type-directed name resolution, type-directed overload 
 * resolution and when generating warning messages.
 *
 * The inference engine can be used in 'undo' mode to implement
 * can-unify predicates used in method overload resolution and trait constraint
 * satisfaction.
 *
 *------------------------------------------------------------------------- *)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Il = Microsoft.Research.AbstractIL.IL 
F#*) 
open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Nums
open Env
open Il (* Abstract IL  *)
open Lib
open Outcome
open Infos
open Printf
open Typrelns

(*-------------------------------------------------------------------------
!* Generate type variables and record them in within the scope of the
 * compilation environment, which currently corresponds to the scope
 * of the constraint resolution carried out by type checking.
 *------------------------------------------------------------------------- *)

   
let new_tv_uniq = let i = ref 0 in fun () -> incr i; !i
let new_anon_or_compgen_inference_tyvar (id,rigid,var,error) = new_typar(rigid,Typar(id,var,true),error,[]) 

let compgen_id = text_to_id0 ("?")
let new_compgen_inference_tyvar (rigid,var,error) = new_anon_or_compgen_inference_tyvar (compgen_id,rigid,var,error) 

let anon_id m = mksyn_id m "?"
let new_anon_inference_tyvar (m,rigid,var,error) = new_anon_or_compgen_inference_tyvar (anon_id m,rigid,var,error) 

let new_inference_tyvar () = new_compgen_inference_tyvar (TyparFlexible,NoStaticReq,false)
let new_error_tyvar () = new_compgen_inference_tyvar (TyparFlexible,NoStaticReq,true)
let new_inference_typ () = TType_var (mk_local_tpref (new_inference_tyvar ()))
let new_error_typ () = TType_var (mk_local_tpref (new_error_tyvar ()))

let new_inference_typs l = map (fun _ -> new_inference_typ ()) l

(* QUERY: should 'rigid' ever really be 'true'? We set this when we know *)
(* we are going to have to generalize a typar, e.g. when implementing a *)
(* abstract generic method slot. But we later check the generalization *)
(* condition anyway, so we could get away with a non-rigid typar. This *)
(* would sort of be cleaner, though give errors later. *)
let freshen_and_fixup_typars amap m rigid fctps tinst tpsorig = 
    let copy_tyvar tp =  new_compgen_inference_tyvar (rigid,static_req_of_typar tp,false) in
    let tps = map copy_tyvar tpsorig in 
    let renaming,tinst = fixup_new_typars amap m fctps tinst tpsorig tps in 
    tps,renaming,tinst

let new_tinst amap m tpsorig = freshen_and_fixup_typars amap m TyparFlexible [] [] tpsorig 
let new_minst amap m fctps tinst tpsorig = freshen_and_fixup_typars amap m TyparFlexible fctps tinst tpsorig 

let freshen_tps amap m tpsorig = 
    let _,renaming,tptys = new_tinst amap m tpsorig in 
    tptys

let formal_tctps_of_minfo g minfo = 
    match minfo with 
    | ILMeth(ilminfo) -> ilminfo |> tinfo_of_il_minfo |> formal_tctps_of_il_tinfo
    | FSMeth(typ,vref) -> 
        let ttps,mtps,rty,tinst = dest_fs_minfo g (typ,vref) in
        ttps
    | DefaultStructCtor typ -> 
        typars_of_tcref (tcref_of_stripped_typ typ)

let freshen_minfo g amap m minfo =
    let tinst = tinst_of_stripped_typ (typ_of_minfo minfo) in 
    if debug then dprintf4 "freshen_minfo @ %a, typ = %s, #mtps = %d\n" output_range m ((DebugPrint.showType (typ_of_minfo minfo))) (length (formal_mtps_of_minfo g minfo));
    let _,renaming,tptys = new_minst amap m (formal_tctps_of_minfo g minfo) tinst (formal_mtps_of_minfo g minfo) in 
    tptys

(*-------------------------------------------------------------------------
!* Unification of types: solve/record equality constraints
 * Subsumption of types: solve/record subtyping constraints
 *------------------------------------------------------------------------- *)

exception ConstraintSolverTupleDiffLengths of displayEnv * Tast.typ list * Tast.typ list * range  * range 
exception ConstraintSolverInfiniteTypes of displayEnv * Tast.typ * Tast.typ * range * range
exception ConstraintSolverTypesNotInEqualityRelation of displayEnv * Tast.typ * Tast.typ * range  * range 
exception ConstraintSolverArrayKindMismatch of displayEnv * Tast.typ * Tast.typ * range  * range 
exception ConstraintSolverTypesNotInSubsumptionRelation of displayEnv * Tast.typ * Tast.typ * range  * range 
exception ConstraintSolverMissingConstraint of displayEnv * Tast.typar_spec * Tast.typar_constraint * range  * range 
exception ConstraintSolverError of string * range * range
exception ConstraintSolverRelatedInformation of string option * range * exn 

exception ErrorFromApplyingDefault of Env.tcGlobals * displayEnv * Tast.local_typar_ref * Tast.typ * error * range
exception ErrorFromAddingTypeEquation of Env.tcGlobals * displayEnv * Tast.typ * Tast.typ * error * range
exception ErrorsFromAddingSubsumptionConstraint of Env.tcGlobals * displayEnv * Tast.typ * Tast.typ * error * range
exception ErrorFromAddingConstraint of  displayEnv * error * range

exception UnresolvedOverloading of displayEnv * error list * error list * string * range
exception PossibleOverload of displayEnv * string * range

let possibleOverloads  g amap m denv calledMethGroup = 
    calledMethGroup |> List.map (fun cmeth -> 
        let minfo = minfo_of_cmeth cmeth in 
        PossibleOverload(denv,string_of_minfo g amap m denv minfo,m))


type constraint_solver_state = 
    { 
      css_g: Env.tcGlobals;
      css_amap: Import.importMap; 
      (** This table stores all trait constraints, indexed by free type variable. *)
      (** That is, there will be one entry in this table for each free type variable in *)
      (** each outstanding trait constraint. Constraints are removed from the table and resolved *)
      (** each time a solution to an index variable is found. *)
      mutable css_cxs: (int, (Tast.trait_constraint_info * range)) Hashtbl.t;
    }


type constraint_solver_env = 
    { 
      cs_css: constraint_solver_state;
      cs_m: range;
      cs_aenv: type_equiv_env;
      cs_denv : displayEnv
    }

let mk_csenv css m denv = 
    { cs_css=css;
      cs_m=m;
      cs_aenv=tyeq_env_empty; 
      cs_denv = denv }


(*-------------------------------------------------------------------------
!* Occurs check
 *------------------------------------------------------------------------- *)

let rec occurs un ty = 
    if verbose then  dprintf0 "--> occurs...\n";
    (* REVIEW: don't strip abbreviations here *)
    match strip_tpeqns_and_tcabbrevs ty with 
    | TType_app (_,l) | TType_tuple l -> exists (occurs un) l
    | TType_fun (d,r) -> occurs un d || occurs un r
    | TType_var r   ->  typar_ref_eq un r 
    | TType_forall (tp,tau) -> occurs un tau
    | _ -> false 


(*-------------------------------------------------------------------------
!* Predicates on types
 *------------------------------------------------------------------------- *)

let signedIntegralType  g ty =
    type_equiv g g.sbyte_ty ty || 
    type_equiv g g.int16_ty ty || 
    type_equiv g g.int32_ty ty || 
    type_equiv g g.nativeint_ty ty || 
    type_equiv g g.int64_ty ty 

let unsignedIntegralType  g ty =
    type_equiv g g.byte_ty ty || 
    type_equiv g g.uint16_ty ty || 
    type_equiv g g.uint32_ty ty || 
    type_equiv g g.unativeint_ty ty || 
    type_equiv g g.uint64_ty ty 

let rec integralType g ty =
    signedIntegralType g ty || 
    unsignedIntegralType g ty || 
    (is_enum_typ ty && integralType g (underlying_typ_of_enum_typ g ty))
    
let stringType g ty =
    type_equiv g g.string_ty ty 
let charType g ty =
    type_equiv g g.char_ty ty 
let  fpType g ty =
    type_equiv g g.float_ty ty || 
    type_equiv g g.float32_ty ty  
let decimalType g ty = 
    type_equiv g g.decimal_ty ty 
let numericType g ty = integralType g ty || fpType g ty 

let isArrayTypeWithIndexer g ty = 
    (if g.typeCheckerConfiguredToAssumeErasureOfGenerics then is_compat_array_typ g ty else is_any_array_typ g ty)

let isArrayTypeWithSpecificIndexer g n ty = 
    isArrayTypeWithIndexer g ty
    && (n = rank_of_any_array_typ g ty)

let isArrayTypeWithSlice g n ty = 
    (if g.typeCheckerConfiguredToAssumeErasureOfGenerics then is_compat_array_typ g ty else is_any_array_typ g ty) && (rank_of_any_array_typ g ty = n)

let isArrayKindMismatch g tc1 l1 tc2 l2 = 
    g.tcref_eq tc1 g.il_arr1_tcr &&
    g.tcref_eq g.array_tcr tc2 && 
    length l1 = 1 && 
    length l2 = 1 && 
    type_equiv g (hd l1) (hd l2)  

let sameHeadType g ty1 ty2 = 
  is_stripped_tyapp_typ ty1 && is_stripped_tyapp_typ ty2 &&
  g.tcref_eq (tcref_of_stripped_typ ty1) (tcref_of_stripped_typ ty2)

let existsSameHeadTypeInHeirarchy g amap m typeToSearchFrom typeWithToLookFor = 
   exists_in_entire_hierarchy_of_typ (sameHeadType g typeWithToLookFor)  g amap m typeToSearchFrom
  

type trait_constraint_solution = 
    | TTraitUnsolved
    | TTraitBuiltIn
    | TTraitSolved of meth_info
    
(*-------------------------------------------------------------------------
!* Run the constraint solver with undo (used during method overload resolution)
 *------------------------------------------------------------------------- *)

type trace = (unit -> unit) list ref

type optionalTrace = 
    | NoTrace
    | Trace of trace
    
let newTrace () =  (ref [])
let undoTrace trace =   iter (fun a -> a ()) !trace
let saveOnPreviousTrace trace1 trace2 = trace1 := !trace1 @ !trace2

let isNoTrace = function NoTrace -> true | Trace _ -> false


let collectThenUndo f = 
    let trace = newTrace() in 
    let res = f trace in 
    undoTrace trace; 
    res

let checkThenUndo f = collectThenUndo f |> gaveNoErrors 

let filterEachWithUndo f meths = 
    filter (fun calledMeth -> checkThenUndo (fun trace -> f trace calledMeth)) meths


(*-------------------------------------------------------------------------
!* Solve
 *------------------------------------------------------------------------- *)

exception NonRigidTypar of string * range
exception LocallyAbortUnificationThatLosesAbbrevs 

  
(** Add the constraint "ty1 = ty" to the constraint problem, where ty1 is a type variable. *)
(** Propagate all effects of adding this constraint, e.g. to solve other variables *)
let rec solveTyparEqualsTyp ndeep csenv m2 trace ty1 ty =
    if verbose then dprintf1 "--> solveTyparEqualsTyp...%s\n" ("ty1 = "^Layout.showL(typeL ty1)^", ty = "^Layout.showL(typeL ty));
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    match ty1 with 
    | TType_var r  ->

      (* The types may still be equivalent due to abbreviations, which we are trying not to eliminate *)
      if type_equiv csenv.cs_css.css_g ty1 ty then completeD else

      (* The famous 'occurs' check to catch things like 'a = list<'a> *)
      if occurs r ty then errorD (ConstraintSolverInfiniteTypes(denv,ty1,ty,m,m2)) else

      (* Check if the type variable came from a user type annotation. If it is being instantiated to a non-variable *)
      (* type then give a warning. Note: warn _and_ continue! *)
      (if rigid_of_tpref r = TyparWarnIfNotRigid && not (is_typar_ty ty) then 
          
          (* NOTE: we format the error eagerly to make sure the type variable prints as a type variable *)
          let _,(ty1,ty),tpcs = PrettyTypes.prettify2 (ty1,ty) in 
          let msg = 
              if is_typar_ty ty1 && compgen_of_typar (dest_typar_typ ty1) && rigid_of_typar  (dest_typar_typ ty1) = TyparWarnIfNotRigid then
                  bufs (fun buf -> Printf.bprintf buf "This construct causes code to be less generic than indicated by its type annotations. The type parameter implied by the use of a '#' or other flexibility annotation at or near '%a' has been constrained to be type '%s'" boutput_range (range_of_typar (dest_typar_typ ty1))  (NicePrint.string_of_typ denv ty))
              else
                  bufs (fun buf -> Printf.bprintf buf "This construct causes code to be less generic than indicated by the type annotations. The type variable %s has been constrained to be type '%s'" (NicePrint.string_of_typ denv ty1)  (NicePrint.string_of_typ denv ty)) in
          warnD(NonRigidTypar(msg,csenv.cs_m  )) 
       else completeD) ++ (fun () ->

      (* Record the solution before we solve the constraints, since *)
      (* We may need to make use of the equation when solving the constraints. *)
      (* Record a entry in the undo trace if one is provided *)
      let tpdata = derefd_local_typar r in 
      begin match trace with 
      | NoTrace -> () 
      | Trace actions -> actions := (fun () -> tpdata.typar_solution <- TType_unknown) :: !actions
      end;
      tpdata.typar_solution <- ty;
      
  (*   dprintf4 "setting typar %d to type %s at %a\n" (stamp_of_typar r) ((DebugPrint.showType ty)) output_range m; *)

      (* Only solve constraints if this is not an error var *)
      if from_error_of_typar r then completeD else
      
      (* Check to see if this type variable is relevant to any trait constraints. *)
      (* If so, r-esolve the relevant constraints. *)
      (if Hashtbl.mem csenv.cs_css.css_cxs  (stamp_of_typar r) then 
           repeatWhileD (fun () -> solveRelevantMemberConstraintsForTypar csenv false trace r)
       else 
           completeD) ++ (fun _ ->
      
      (* Re-solve the other constraints associated with this type variable *)
      solveTypMeetsTyparConstraints ndeep csenv m2 trace ty (static_req_of_tpref r,constraints_of_tpref r)))
      
    | _ -> failwith "solveTyparEqualsTyp"


(** Given a type 'ty' and a set of constraints on that type, solve those constraints. *)
and solveTypMeetsTyparConstraints ndeep csenv m2 trace ty (sreq,cs) =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    (* Propagate static requirements from 'tp' to 'ty' *)
    solveTypStaticReq csenv trace sreq ty ++ (fun () -> 
    
    (* Solve constraints on 'tp' w.r.t. 'ty' *)
    cs |> iterD (function
      | TTyparDefaultsToType (priority,dty,m) -> 
          if not (is_typar_ty ty) or type_equiv g ty dty then completeD else
          recordConstraint ndeep csenv m2 trace (dest_typar_typ ty)  (TTyparDefaultsToType(priority,dty,m))
          
      | TTyparSupportsNull m2               -> solveTypSupportsNull               ndeep csenv m2 trace ty
      | TTyparIsEnum(underlying, m2)        -> solveTypIsEnum                     ndeep csenv m2 trace ty underlying
      | TTyparIsDelegate(aty,bty, m2)       -> solveTypIsDelegate                 ndeep csenv m2 trace ty aty bty
      | TTyparIsNotNullableValueType m2     -> solveTypIsNonNullableValueType     ndeep csenv m2 trace ty
      | TTyparIsReferenceType m2            -> solveTypIsReferenceType            ndeep csenv m2 trace ty
      | TTyparRequiresDefaultConstructor m2 -> solveTypRequiresDefaultConstructor ndeep csenv m2 trace ty
      | TTyparSimpleChoice(tys,m2)          -> solveTypChoice                     ndeep csenv m2 trace ty tys
      | TTyparCoercesToType(TTyparSubtypeConstraintFromFS(ty2),m2) -> solveTypSubsumesTyp ndeep csenv m2 trace ty2 ty
      | TTyparCoercesToType(TTyparSubtypeConstraintFromIL(_),_) -> 
          warnD(InternalError("solveTypMeetsTyparConstraints ndeep: unexpected TTyparSubtypeConstraintFromIL",m))
      | TTyparMayResolveMemberConstraint(traitInfo,m2) -> 
          solveMemberConstraint false ndeep csenv m2 trace traitInfo ++ (fun _ -> completeD) 
    ))  

        
(** Add the constraint "ty1 = ty2" to the constraint problem. *)
(** Propagate all effects of adding this constraint, e.g. to solve type variables *)
(** May raise LocallyAbortUnificationThatLosesAbbrevs should we fail, which effectively implements a localized 'goto' *)
and solveTypEqualsTyp ndeep csenv m2 trace ty1 ty2 = 
    if verbose then  dprintf2 "solveTypEqualsTyp ndeep @ %a\n" output_range csenv.cs_m;
    let ndeep = ndeep + 1 in 
    let aenv = csenv.cs_aenv in 
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    if ndeep > 100 then error(InternalError("recursive class hierarchy (detected in solveTypEqualsTyp), ty1 = "^Layout.showL(typeL ty1),m)) else
    if ty1 =!= ty2 then completeD else
    let canShortcut = (isNoTrace trace) in
    let sty1 = strip_tpeqns_and_tcabbrevsA canShortcut ty1 in 
    let sty2 = strip_tpeqns_and_tcabbrevsA canShortcut ty2 in
    match sty1, sty2 with 
    (* type vars inside forall-types may be alpha-equivalent *)
    | TType_var tp1, TType_var tp2 when  typar_ref_eq tp1 tp2 || (tpmap_mem tp1 aenv.ae_typars && type_equiv g (tpmap_find tp1 aenv.ae_typars) ty2) -> completeD
    (* Prefer to unify away compiler generated type vars *)
    | TType_var r, TType_var _  when (compgen_of_tpref r) && (rigid_of_tpref r <> TyparRigid) -> solveTyparEqualsTyp ndeep csenv m2 trace sty1 ty2
    | TType_var _, TType_var r  when (compgen_of_tpref r) && (rigid_of_tpref r <> TyparRigid) -> solveTyparEqualsTyp ndeep csenv m2 trace sty2 ty1
    (* Prefer to unify away non-error vars - gives better error recovery since we keep *)
    (* error vars lying around, and can avoid giving erros about illegal polymorphism *)
    (* if they occur *)
    | TType_var r, TType_var _ when not (from_error_of_typar r) && (rigid_of_tpref r <> TyparRigid) -> solveTyparEqualsTyp ndeep csenv m2 trace sty1 ty2
    | TType_var _, TType_var r when not (from_error_of_typar r) && (rigid_of_tpref r <> TyparRigid) -> solveTyparEqualsTyp ndeep csenv m2 trace sty2 ty1
    | TType_var r, _ when (rigid_of_tpref r <> TyparRigid) -> solveTyparEqualsTyp ndeep csenv m2 trace sty1 ty2
    | _, TType_var r when (rigid_of_tpref r <> TyparRigid) -> solveTyparEqualsTyp ndeep csenv m2 trace sty2 ty1
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when g.tcref_eq tc1 tc2  -> solveTypEqualsTypEqns ndeep csenv m2 trace l1 l2
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when isArrayKindMismatch g tc1 l1 tc2 l2 -> errorD (ConstraintSolverArrayKindMismatch(denv,ty1,ty2,m,m2))
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when isArrayKindMismatch g tc2 l2 tc1 l1 -> errorD (ConstraintSolverArrayKindMismatch(denv,ty2,ty1,m,m2))
    | TType_app (tc1,_)   ,TType_app (tc2,_)   ->  raise (LocallyAbortUnificationThatLosesAbbrevs)
    | TType_tuple l1      ,TType_tuple l2      -> solveTypEqualsTypEqns ndeep csenv m2 trace l1 l2
    | TType_fun (d1,r1)   ,TType_fun (d2,r2)   -> solveFunTypEqn ndeep csenv m2 trace d1 d2 r1 r2
    | TType_unknown    , _                     -> failwith "unif: naked unknown"
    | _                , TType_unknown         -> failwith "unif: naked unknown"
    | TType_forall(tps1,rty1), TType_forall(tps2,rty2) -> 
        if (length tps1 <> length tps2) then raise LocallyAbortUnificationThatLosesAbbrevs;
        let aenv = bind_tyeq_env_typars tps1 tps2 aenv in 
        let csenv = {csenv with cs_aenv = aenv }in 
        if not (typar_decls_aequiv g aenv tps1 tps2) then raise LocallyAbortUnificationThatLosesAbbrevs;
        solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty1 rty2 
    | _  -> raise LocallyAbortUnificationThatLosesAbbrevs

and solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace ty1 ty2 = 
   let denv = csenv.cs_denv in 
   
   (* The exception LocallyAbortUnificationThatLosesAbbrevs is used to back out of back out of expansions of type abbreviations *)
   (* to give improved error messages. *)
   try solveTypEqualsTyp ndeep csenv m2 trace ty1 ty2 
   with LocallyAbortUnificationThatLosesAbbrevs -> errorD(ConstraintSolverTypesNotInEqualityRelation(denv,ty1,ty2,csenv.cs_m,m2))

and solveTypEqualsTypEqns ndeep csenv m2 trace l1 l2 = 
   let denv = csenv.cs_denv in 
   if length l1 <> length l2 then errorD(ConstraintSolverTupleDiffLengths(denv,l1,l2,csenv.cs_m,m2)) else
   iter2D (solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace) l1 l2 

and solveFunTypEqn ndeep csenv m2 trace d1 d2 r1 r2 = 
    solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace d1 d2 ++ (fun () -> 
    solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace r1 r2)

and solveTypSubsumesTyp ndeep csenv m2 trace ty1 ty2 = 
    (* 'a :> obj ---> <solved> *)
    if verbose then dprintf1 "--> solveTypSubsumesTyp...%s\n" ("ty1 = "^Layout.showL(typeL ty1)^", ty2 = "^Layout.showL(typeL ty2));
    let ndeep = ndeep + 1 in 
    let g = csenv.cs_css.css_g in 
    let amap = csenv.cs_css.css_amap in 
    let aenv = csenv.cs_aenv in 
    let denv = csenv.cs_denv in 
    let m = csenv.cs_m in 
    if ndeep > 100 then error(Error("recursive class hierarchy (detected in solveTypSubsumesTyp), ty1 = "^Layout.showL(typeL ty1),m));
    if (type_equiv g ty1 g.obj_ty) then completeD else
    (* note: values with TType_forall may not be passed to .NET *)
    let canShortcut = (isNoTrace trace) in
    let sty1 = strip_tpeqns_and_tcabbrevsA canShortcut ty1 in 
    let sty2 = strip_tpeqns_and_tcabbrevsA canShortcut ty2 in
    match sty1, sty2 with 
    | TType_var tp1, _ 
        when  tpmap_mem tp1 aenv.ae_typars -> 
        solveTypSubsumesTyp ndeep csenv m2 trace (tpmap_find tp1 aenv.ae_typars) ty2 
        
    | TType_var r1, TType_var r2 when typar_ref_eq r1 r2 -> completeD
    | _, TType_var r (* when not (rigid_of_tpref r) *) -> solveTyparSubtypeOfType ndeep csenv m2 trace r ty1
    | TType_var r , _ (* | _, TType_var r *)  ->  solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace ty1 ty2
    | TType_tuple l1    ,TType_tuple l2     -> solveTypEqualsTypEqns ndeep csenv m2 trace l1 l2 (* nb. can unify since no variance *)
    | TType_fun (d1,r1)  ,TType_fun (d2,r2)   -> solveFunTypEqn ndeep csenv m2 trace d1 d2 r1 r2 (* nb. can unify since no variance *)
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when g.tcref_eq tc1 tc2  -> 
        solveTypEqualsTypEqns ndeep csenv m2 trace l1 l2
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when isArrayKindMismatch g tc1 l1 tc2 l2 ->
        errorD (ConstraintSolverArrayKindMismatch(denv,ty1,ty2,m,m2))
    | TType_app (tc1,l1)  ,TType_app (tc2,l2) when isArrayKindMismatch g tc2 l2 tc1 l1 ->
        errorD (ConstraintSolverArrayKindMismatch(denv,ty2,ty1,m,m2))
    | _ ->  
        (* by now we know the type is not a variable type *)

        (* 'a :> obj ---> <solved> *)
        if (type_equiv g ty1 g.obj_ty) then completeD else

        (* 'a[] :> IList<'b>   ---> 'a =!= 'b  *)
        (* 'a[] :> ICollection<'b>   ---> 'a =!= 'b  *)
        (* 'a[] :> IEnumerable<'b>   ---> 'a =!= 'b  *)
        (* Note we don't support co-variance on array types nor *)
        (* at these special .NET conversions. *)
        if 
            (g.typeCheckerConfiguredToAssumeV20Library && 
             is_il_arr1_typ g ty2 &&  
             is_stripped_tyapp_typ ty1 && 
             (let tcr1 = tcref_of_stripped_typ ty1 in 
              g.tcref_eq tcr1 g.tcref_System_Collections_Generic_IList || 
              g.tcref_eq tcr1 g.tcref_System_Collections_Generic_ICollection || 
              g.tcref_eq tcr1 g.tcref_System_Collections_Generic_IEnumerable)) then

          let tcref,tinst = dest_stripped_tyapp_typ ty1 in 
          if length tinst <> 1 then error(InternalError("dest_il_arr1_typ",m));
          let ty1arg = (hd tinst) in 
          let ty2arg = dest_il_arr1_typ g ty2 in 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace ty1arg  ty2arg

        (* D<inst> :> Head<_> --> C<inst'> :> Head<_> for the *)
        (* first interface or super-class C supported by D which *)
        (* may feasibly convert to Head. *)

        else 
            match (find_unique_feasible_supertype g amap m ty1 ty2) with 
           | None -> errorD(ConstraintSolverTypesNotInSubsumptionRelation(denv,ty1,ty2,m,m2))
           | Some t -> solveTypSubsumesTyp ndeep csenv m2 trace ty1 t

(*-------------------------------------------------------------------------
!* Solve and record non-equality constraints
 *------------------------------------------------------------------------- *)

and transact_static_req csenv trace tpr req = 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    if (rigid_of_tpref tpr = TyparRigid && static_req_of_typar tpr <> req) then 
        errorD(ConstraintSolverError("The declared type parameter '"^name_of_tpref tpr^" cannot be used here since the type parameter cannot be resolved at compile time",m,m)) 
    else
        let tpdata = derefd_local_typar tpr in 
        let orig = static_req_of_typar tpr in 
        begin match trace with 
        | NoTrace -> () 
        | Trace actions -> actions := (fun () -> set_static_req_of_tpdata tpdata orig) :: !actions
        end;
        set_static_req_of_tpdata tpdata req;
        completeD

and solveTypStaticReq csenv trace req ty =
    match req with 
    | NoStaticReq -> completeD
    | CompleteStaticReq -> 
        (* requires that a type be completely known at compile time *)
        (free_in_type ty).free_loctypars |> Zset.elements |> iterD (fun tpr -> transact_static_req csenv trace tpr CompleteStaticReq)
    | HeadTypeStaticReq -> 
        (* requires that a type constructor be known at compile time *)
        if (is_typar_ty ty) then 
          let tpr = dest_typar_typ ty in 
          let orig = (static_req_of_tpref tpr) in
          let req2 = joinTyparStaticReq req orig in
          if orig <> req2 then transact_static_req csenv trace tpr req2 else completeD
        else completeD
      
and solveTyparSubtypeOfType ndeep csenv m2 trace tp ty1 = 
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    if (type_equiv g ty1 g.obj_ty) then completeD
    else if type_equiv g ty1 (TType_var tp) then completeD
    else if is_sealed_typ g ty1 then 
        solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace (TType_var tp) ty1
    else 
        recordConstraint ndeep csenv m2 trace tp  (TTyparCoercesToType(TTyparSubtypeConstraintFromFS(ty1),m))

(* Do a bunch of fakery to pretend that primitive types have certain members. *)
(* We pretend int and other types support a number of operators.  In the actual IL for mscorlib they *)
(* don't, however the type-directed static optimization rules in the library code that makes use of this *)
(* will deal with the problem. *)
and solveMemberConstraint canon ndeep csenv m2 trace (TTrait(tys,nm,memFlags,argtys,rty)) : trait_constraint_solution operationResult =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let amap = csenv.cs_css.css_amap in 
    let aenv = csenv.cs_aenv in 
    let denv = csenv.cs_denv in 
    if verbose then dprintf1 "-----------------------------\nResolve trait for %s\n" nm;

    (* Remove duplicates from the set of types in the support *)
    let tys = gen_setify (type_aequiv g aenv) tys in
    (* Rebuild the trait infor after removing duplicates *)
    let traitInfo = (TTrait(tys,nm,memFlags,argtys,rty)) in 
    
    
    begin
        if memFlags.memFlagsInstance then 
        match tys, argtys with 
        | [ty], (h :: t) -> solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace h ty
        | _ -> errorD (ConstraintSolverError("Expected arguments to an instance member", m,m2))
        else completeD
    end ++ (fun () -> 
    
    (* Trait calls are only supported on pseudo type (variables) *)
    (* Do this after doing the types - better error messages that way *)
    tys |> iterD (solveTypStaticReq csenv trace HeadTypeStaticReq)) ++ (fun () -> 
    
    let argtys = if memFlags.memFlagsInstance then List.tl argtys else argtys in 

    let minfos = relevantMethodsForTrait canon csenv nm traitInfo in

    if verbose then minfos |> List.iter (fun minfo -> dprintf1 "Possible overload: %s\n" (string_of_minfo g amap m denv minfo));
        
    begin match minfos,tys,memFlags.memFlagsInstance,nm,argtys with 
      | [],_,false,("op_Addition" | "op_Subtraction" | "op_Multiply" | "op_Division" | "op_Modulus"),[argty1;argty2] 
          when    (numericType g argty1 || (nm = "op_Addition" && stringType g argty1)) && (canon || not (is_typar_ty argty2))
               || (numericType g argty2 || (nm = "op_Addition" && stringType g argty2)) && (canon || not (is_typar_ty argty1)) ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 argty1 ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty1 ++ (fun () -> 
          resultD TTraitBuiltIn))

      | [],_,false,("op_Range"),[argty1;argty2] 
          when    (numericType g argty1 || charType g argty1) && (canon || not (is_typar_ty argty2))
               || (numericType g argty2 || charType g argty2) && (canon || not (is_typar_ty argty1)) ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 argty1 ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty (mk_seq_ty g argty1) ++ (fun () -> 
          (if fpType g argty1 || decimalType g argty1 then warnD(Experimental("Floating point ranges are experimental and may be deprecated in a future release. Consider using an integer range instead, converting to floating point using 'float' or 'float32'",m)) else completeD) ++ (fun () -> 
          resultD TTraitBuiltIn)))

      | [],_,false,("op_RangeStep"),[argty1;argty2;argty3] 
          when numericType g argty1 || 
               numericType g argty2 || 
               numericType g argty3 ->
          
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 argty1 ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty3 argty1 ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty (mk_seq_ty g argty1) ++ (fun () -> 
          (if fpType g argty1 then warnD(Experimental("Floating point ranges are experimental and may be deprecated in a future release. Consider using an integer range instead, converting to floating point using 'float' or 'float32'",m)) else completeD)++ (fun () -> 
          resultD TTraitBuiltIn))))

      (* We pretend for uniformity that string and 1D array types have a method called 'GetSlice' *)
      | [], [ty],true,"GetSlice",[argty1;argty2] 
          when (stringType g ty) or (isArrayTypeWithSlice g 1 ty) ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty1 (mk_option_ty g g.int_ty) ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 (mk_option_ty g g.int_ty) ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty ty ++ (fun () -> 
          resultD TTraitBuiltIn)));

      (* We pretend for uniformity that the 2D array types have a method called 'GetSlice2D' *)
      | [], [ty],true,"GetSlice2D",[argty1;argty2;argty3;argty4] 
          when (isArrayTypeWithSlice g 2 ty) ->
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty1 (mk_option_ty g g.int_ty) ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 (mk_option_ty g g.int_ty) ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty3 (mk_option_ty g g.int_ty) ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty4 (mk_option_ty g g.int_ty) ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty ty ++ (fun () -> 
          resultD TTraitBuiltIn)))))

  (*
      (* We could pretend for uniformity that the numeric types have a static property called Zero and One *)
      | false,("get_Zero" | "get_One"),[] 
          when numericType g ty ->
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty ty
  *)        
      (* We pretend for uniformity that the 'string' and 'array' types have an indexer property called 'Item' *)
      | [], [ty],true,("get_Item"),[argty1] 
          when stringType g ty ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty1 g.int_ty ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty g.char_ty ++ (fun () -> 
          resultD TTraitBuiltIn))
          (* warnD(OCamlCompatibility("An use of the operator 'expr.[idx]' involved a lookup on an object of indeterminate type. This is deprecated in F# unless OCaml-comaptibility is enabled. Consider adding further type constraints",m) *)

      | [], [ty],true,("get_Item"),argtys
          when isArrayTypeWithSpecificIndexer g (length argtys) ty ->
          
          (argtys |> iterD (fun argty -> solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty g.int_ty)) ++ (fun () -> 
          let ety = dest_any_array_typ g ty in 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty ety ++ (fun () -> 
          resultD TTraitBuiltIn))

      | [], [ty],true,("set_Item"),argtys
          when isArrayTypeWithSpecificIndexer g (length argtys - 1) ty ->
          
          let argtys,ety = frontAndBack argtys in 
          (argtys |> iterD (fun argty -> solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty g.int_ty)) ++ (fun () -> 
          let etys = dest_any_array_typ g ty in 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace ety etys ++ (fun () -> 
          resultD TTraitBuiltIn))

      | [], _,false,("op_Exponentiation"),[argty1;argty2] 
          when    type_equiv g g.float_ty argty1 
               || type_equiv g g.float_ty argty2 ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 g.float_ty ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty1 ++ (fun () -> 
          resultD TTraitBuiltIn))

      | [],_,false,("op_BitwiseAnd" | "op_BitwiseOr" | "op_ExclusiveOr"),[argty1;argty2] 
          when    (integralType g argty1 || (is_flag_enum_typ g argty1)) && (canon || not (is_typar_ty argty2))
               || (integralType g argty2 || (is_flag_enum_typ g argty2)) && (canon || not (is_typar_ty argty1)) ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 argty1 ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty1 ++ (fun () -> 
          resultD TTraitBuiltIn));

      | [], _,false,("op_LeftShift" | "op_RightShift"),[argty1;argty2] 
          when    integralType g argty1  ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 g.int_ty ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty1 ++ (fun () -> 
          resultD TTraitBuiltIn))

      | _,_,false,("op_UnaryPlus"),[argty] 
          when numericType g argty  || decimalType g argty ->  

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty ++ (fun () -> 
          resultD TTraitBuiltIn)

      | _,_,false,("op_Increment" | "op_Decrement"),[argty] 
          when integralType g argty   ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty ++ (fun () -> 
          resultD TTraitBuiltIn)

      | _,_,false,("op_UnaryNegation"),[argty] 
          when signedIntegralType g argty || fpType g argty || decimalType g argty  ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty ++ (fun () -> 
          resultD TTraitBuiltIn)

      | _,_,false,("Sign"),[argty] 
          when signedIntegralType g argty || fpType g argty || decimalType g argty ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty g.int32_ty ++ (fun () -> 
          resultD TTraitBuiltIn)

      | _,_,false,("op_LogicalNot" | "op_OnesComplement"),[argty] 
          when integralType g argty  ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty ++ (fun () -> 
          resultD TTraitBuiltIn)

      | _,_,false,("Abs"),[argty] 
          when signedIntegralType g argty || fpType g argty ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty ++ (fun () -> 
          resultD TTraitBuiltIn)

      | _,_,false,("Sin" | "Cos" | "Tan" | "Sinh" | "Cosh" | "Tanh" | "Atan" | "Acos" | "Asin" | "Exp" | "Ceiling" | "Floor" | "Round" | "Log10" | "Log" | "Sqrt"),[argty] 
          when fpType g argty ->

          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty ++ (fun () -> 
          resultD TTraitBuiltIn)

      | _,_,false,("ToChar" | "ToByte" | "ToSByte" | "ToInt16" | "ToUInt16" | "ToInt32" | "ToUInt32" | "ToInt64" | "ToUInt64" | "ToSingle" | "ToDouble" | "ToDecimal" | "ToUIntPtr" | "ToIntPtr"),[argty] 
          when numericType g argty || stringType g argty || charType g argty ->

          resultD TTraitBuiltIn

      | _,_,false,("ToDecimal"),[argty] 
          when numericType g argty || stringType g argty ->

          resultD TTraitBuiltIn

      | _,_,false,("ToUIntPtr" | "ToIntPtr"),[argty] 
          when numericType g argty || charType g argty -> (* note: IntPtr and UIntPtr are different, they do not support .Parse() from string *)

          resultD TTraitBuiltIn

      | [],_,false,("Atan2" | "Pow"),[argty1; argty2] 
          when fpType g argty1 ->
          
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace argty2 argty1 ++ (fun () -> 
          solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty argty1 ++ (fun () -> 
          resultD TTraitBuiltIn))

      | _ -> 

        if List.exists is_fun_ty tys  then 
            errorD (ConstraintSolverError("Expecting a type supporting the operator '"^decompileOpName nm^"' but given a function type. Are you missing an argument to a function?",m,m2)) else

        begin match minfos,tys  with 
        | [],[ty] when not (is_typar_ty ty) ->
            errorD (ConstraintSolverError("The type '"^(NicePrint.pretty_string_of_typ denv ty)^"' does not support any operators named '"^decompileOpName nm^"'",m,m2))

        | _ -> 

            let dummyExpr = mk_unit g m in
            let calledMethGroup = 
                minfos |> map (fun minfo -> 
                    let callerArgs = argtys |> map (fun argty -> CallerArg(argty,m,false,dummyExpr)) in
                    mk_formalCalledMeth g amap m minfo callerArgs) in

            (* dprintf2 "    ---> calling resolveOverloading, nm = %s, ty = '%s'\n" nm (Layout.showL (typeL ty)); *)

            let result,errors = 
                collectThenUndo (fun trace -> resolveOverloading csenv (Trace(trace)) nm (0,0) AccessibleFromEverywhere calledMethGroup (Some (rty,dummyExpr)))   in 

            if verbose then dprintf1 "    <--- called resolveOverloading, ok? = %b\n" (gaveNoErrors errors);
                
            match result with 
            | Some calledMeth -> 
                (* OK, the cosntraint is solved. *)
                (* Re-run without undo to commit the inference equations. Throw errors away *)
                let minfo = minfo_of_cmeth calledMeth in
                if verbose then dprintf1 "    ---> constraint solved, calling resolveOverloading a second time, without undo, minfo = %s\n" (string_of_minfo g amap m denv minfo);
                let _,errors = resolveOverloading csenv trace nm (0,0) AccessibleFromEverywhere calledMethGroup (Some (rty,dummyExpr)) in

                errors ++ (fun () -> 
                    let isInstance = minfo_is_instance minfo in 
                    if isInstance <> memFlags.memFlagsInstance then 
                        errorD(ConstraintSolverError("The type '"^(NicePrint.pretty_string_of_typ denv (typ_of_minfo minfo))^"' has a method '"^decompileOpName nm^"' (full name '"^nm^"'), but the method is"^(if isInstance then " not" else "")^" static",m,m2 ))
                    else if generic_arity_of_minfo g minfo  > 0 then 
                        errorD(ConstraintSolverError("The type '"^(NicePrint.pretty_string_of_typ denv (typ_of_minfo minfo))^"' has a method '"^decompileOpName nm^"' (full name '"^nm^"'), but the method is more generic than the containing type. You may need to constrain the implementation of the method to be in terms of the type parameters of the enclosing type",m,m2))
                    else 
                        minfo_attrib_check g m minfo ++ (fun () -> 
                            resultD (TTraitSolved minfo)))
                  
            | None ->
                    
                    
                let support =  supportOfMemberConstraint traitInfo in 

                (* If there's nothing left to learn then raise the errors *)
                (if canon && isNil(support) then errors  
                (* Otherwise re-record the trait waiting for canonicalization *)
                 else recordMemberConstraint ndeep csenv m2 trace traitInfo) ++ (fun () -> 
                        resultD TTraitUnsolved)
        end     
    end )


and relevantMethodsForTrait canon csenv nm (TTrait(tys,nm,memFlags,argtys,rty) as traitInfo) =
    (* Only consider overload resolution if canonicalizing or all the types are now nominal. *)
    (* That is, don't perform resolution if more nominal information may influence the set of available overloads *)
    if canon || isNil (supportOfMemberConstraint traitInfo) then
        let m = csenv.cs_m in 
        let g = csenv.cs_css.css_g in 
        let amap = csenv.cs_css.css_amap in 
        let minfos = map (intrinsic_minfos_of_typ (Some(nm),DontIncludePrivate) IgnoreOverrides g amap m) tys in
        (* Merge the sets so we don't get the same minfo from each side *)
        (* We merge based on whether minfos use identical metadata or not. *)

        (* REVIEW: It seems feasible there are some vaguely interesting fairly pathological cases where this may cause a loss of distinction *)
        (* between potential overloads because a generic instantiation derived from the left hand type differs *)
        (* to a generic instantiation for an operator based on the right hand type. *)
        
        let minfos = List.fold_left (gen_union_favour_left (minfos_use_identical_definitions g)) (List.hd minfos) (List.tl minfos) in
        minfos
    else 
        []



(** The nominal support of the member constraint *)
and supportOfMemberConstraint (TTrait(tys,nm,memFlags,argtys,rty)) =
    tys |> chooseList (fun ty -> if is_typar_ty ty then Some (dest_typar_typ ty) else  None)
    
(*
    (free_in_types_lr false (tys@argtys@[rty]))
    |> filter (fun tp -> static_req_of_typar tp <> NoStaticReq) 
*)

(** All the typars relevant to the member constraint *)
and freeTyparsOfMemberConstraint (TTrait(tys,nm,memFlags,argtys,rty)) =
    (free_in_types_lr_no_cxs (tys@argtys@[rty]))

(** Re-solve the global constraints involving any of the given type variables. *)
(** Trait constraints can't always be solved using the pessimistic rules. As a result we only canonicalize *)
(** them forcefully prior to generalization. *)
and solveRelevantMemberConstraints csenv canon trace tps =

    repeatWhileD
        (fun () -> 
            tps |> atLeastOneD (fun tp -> 
                (** Normalize the typar *)
                let ty = mk_typar_ty tp in 
                if not (is_typar_ty ty) then resultD false else
                let tp = dest_typar_typ ty in 
                solveRelevantMemberConstraintsForTypar csenv canon trace tp))

and solveRelevantMemberConstraintsForTypar csenv canon trace tp =
    let cxst = csenv.cs_css.css_cxs in 
    let tpn = (stamp_of_typar tp) in
    let cxs = Hashtbl.find_all  cxst tpn in
    if isNil cxs then resultD false else
    begin 
        if verbose then dprintf3 "solveRelevantMemberConstraintsForTypar #cxs = %d, m = %a\n" (List.length cxs) output_range csenv.cs_m;
        cxs |> List.iter (fun _ -> Hashtbl.remove cxst tpn);

        assert (Hashtbl.find_all cxst tpn = []);

        begin match trace with 
        | NoTrace -> () 
        | Trace actions -> actions := (fun () -> List.iter (Hashtbl.add cxst tpn) cxs) :: !actions
        end;

        cxs |> atLeastOneD (fun (traitInfo,m2) -> 
            let csenv = { csenv with cs_m = m2 } in 
            solveMemberConstraint canon 0 csenv m2 trace traitInfo  ++ (fun res -> resultD (res <> TTraitUnsolved)))
    end

and canonicalizeRelevantMemberConstraints csenv trace tps =
    solveRelevantMemberConstraints csenv true trace tps 

  
and recordMemberConstraint ndeep csenv m2 trace traitInfo =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let aenv = csenv.cs_aenv in 


    let cxst = csenv.cs_css.css_cxs in 
    (* Write the constraint into the global table *)
    traitInfo |> freeTyparsOfMemberConstraint |> List.iter (fun tp -> 
        let tpn = (stamp_of_typar tp) in

        let cxs = Hashtbl.find_all  cxst tpn in
        if verbose then dprintf4 "recordMemberConstraint: tpn = %d, #cxs = %d, m = %a\n" tpn (List.length cxs) output_range csenv.cs_m;
        if verbose && List.length cxs > 10 then 
            cxs |> List.iter (fun (cx,_) -> dprintf2 "     --> cx = %s, fvs = %s\n" (Layout.showL (traitL cx)) (Layout.showL (typarsL (freeTyparsOfMemberConstraint cx))));

        if cxs |> List.exists (fun (traitInfo2,_) -> traits_aequiv g aenv traitInfo traitInfo2) 
        then () 
        else (

            begin match trace with 
            | NoTrace -> () 
            | Trace actions -> actions := (fun () -> Hashtbl.remove csenv.cs_css.css_cxs tpn) :: !actions
            end;
            Hashtbl.add csenv.cs_css.css_cxs tpn (traitInfo,m2)
        );
    );
    let support = supportOfMemberConstraint traitInfo in
    support |> iterD (fun tp -> recordConstraint ndeep csenv m2 trace tp (TTyparMayResolveMemberConstraint(traitInfo,m2)))

    
and recordConstraint ndeep csenv m2 trace tp newConstraint  =
    let g = csenv.cs_css.css_g in 
    let aenv = csenv.cs_aenv in 
    let amap = csenv.cs_css.css_amap in 
    let denv = csenv.cs_denv in 
    let m = csenv.cs_m in 
    (* Record a constraint on an inference type variable. *)


    let consistent tpc1 tpc2 =
        match tpc1,tpc2 with           
        | TTyparMayResolveMemberConstraint(TTrait(tys1,nm1,memFlags1,argtys1,rty1),_),
          TTyparMayResolveMemberConstraint(TTrait(tys2,nm2,memFlags2,argtys2,rty2),_)  
              when memFlags1 = memFlags2 &&
                   nm1 = nm2 &&
                   List.length tys1 = List.length tys2 &&
                   List.for_all2 (type_equiv g) tys1 tys2 -> 

                  
                  iter2D (solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace) argtys1 argtys2 ++ (fun () -> 
                      solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace rty1 rty2 ++ (fun () -> 
                         if verbose then dprintf2 "\n-------------\nmerged constraint for %s, tp = %s\n---------\n" nm1 (Layout.showL (typarDeclL tp));
                          completeD))
          
        | TTyparCoercesToType(TTyparSubtypeConstraintFromFS(ty1),_), 
          TTyparCoercesToType(TTyparSubtypeConstraintFromFS(ty2),_) -> 


               (* Record at most one subtype constraint for each head type. *)
               (* That is, we forbid constraints by both I<string> and I<int>. *)
               (* This works because the types on the r.h.s. of subtype *)
               (* constraints are head-types and so any further inferences are equational. *)
              let collect ty = let res = ref [] in iter_entire_hierarchy_of_typ (fun x -> res := x :: !res) g amap m ty; List.rev !res in
              let parents1 = collect ty1 in
              let parents2 = collect ty2 in
              parents1 |> iterD (fun ty1Parent -> 
                 parents2 |> iterD (fun ty2Parent ->  
                     if not (sameHeadType g ty1Parent ty2Parent) then completeD else
                     solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace ty1Parent ty2Parent))

        | TTyparIsEnum (u1,_),
          TTyparIsEnum (u2,m2) ->   
            solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace u1 u2

        | TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,_), _ 
        | _,TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,_) -> 
            warnD(InternalError("recordConstraint: unexpected TTyparSubtypeConstraintFromIL",m))

        | TTyparSupportsNull _,TTyparSupportsNull _  
        | TTyparIsNotNullableValueType _,TTyparIsNotNullableValueType _     
        | TTyparIsReferenceType _,TTyparIsReferenceType _ 
        | TTyparRequiresDefaultConstructor _,TTyparRequiresDefaultConstructor _ 
        | TTyparSimpleChoice (_,_),TTyparSimpleChoice (_,_) -> 
            completeD
            
        | _ -> completeD in

    (* See when one constraint implies implies another. *)
    (* 'a :> ty1  implies 'a :> 'ty2 if the head type name of ty2 (say T2) occurs anywhere in the heirarchy of ty1 *)
    (* If it does occcur, e.g. at instantiation T2<inst2>, then the check above will have enforced that *)
    (* T2<inst2> = ty2 *)
    let implies tpc1 tpc2 = 
            match tpc1,tpc2 with           
            | TTyparMayResolveMemberConstraint(trait1,_),
              TTyparMayResolveMemberConstraint(trait2,_) -> 
                traits_aequiv g aenv trait1 trait2
            | TTyparCoercesToType(TTyparSubtypeConstraintFromFS(ty1),_), 
                TTyparCoercesToType(TTyparSubtypeConstraintFromFS(ty2),_) -> 
                  existsSameHeadTypeInHeirarchy g amap m ty1 ty2
            | TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,_), _ 
            | _,TTyparCoercesToType(TTyparSubtypeConstraintFromIL _,_) -> 
                warning(InternalError("recordConstraint ndeep: unexpected TTyparSubtypeConstraintFromIL",m));
                false
            | TTyparIsEnum(u1,_),TTyparIsEnum(u2,_) -> type_equiv g u1 u2
            | TTyparSupportsNull _,TTyparSupportsNull _  
            | TTyparIsNotNullableValueType _,TTyparIsNotNullableValueType _     
            | TTyparIsReferenceType _,TTyparIsReferenceType _ 
            | TTyparRequiresDefaultConstructor _,TTyparRequiresDefaultConstructor _ -> true
            | TTyparSimpleChoice (tys1,_),TTyparSimpleChoice (tys2,_) -> gen_subset_of (type_equiv g) tys1 tys2
            | TTyparDefaultsToType (priority1,dty1,_), TTyparDefaultsToType (priority2,dty2,m) -> 
                 (priority1 = priority2) && type_equiv g dty1 dty2
            | _ -> false in

        
    
    (* First ensure constraint conforms with existing constraints *)
    (* NOTE: QUADRATIC *)
    let existingConstraints = constraints_of_tpref tp in 
    if verbose && length existingConstraints > 10 then 
            dprintf1 "    (prior) tp = %s\n" (Layout.showL (typarDeclL tp));

    let allCxs = newConstraint :: List.rev existingConstraints in
    begin 
        let rec enforceMutualConsistency i cxs = 
            match cxs with 
            | [] ->  completeD
            | cx :: rest -> iteriD (fun j cx2 -> if i = j then completeD else consistent cx cx2) allCxs ++ (fun () -> enforceMutualConsistency (i+1) rest) in

        enforceMutualConsistency 0 allCxs 
    end ++ (fun ()  ->
    
    let impliedByExistingConstraints = existingConstraints |> List.exists (fun tpc2 -> implies tpc2 newConstraint)  in
    if verbose then dprintf1 "  impliedByExistingConstraints?  %b\n" impliedByExistingConstraints;
    
    if not impliedByExistingConstraints && (rigid_of_typar tp = TyparRigid) then 
            errorD (ConstraintSolverMissingConstraint(denv,tp,newConstraint,m,m2)) 

    else if impliedByExistingConstraints then 
        (if verbose && length existingConstraints > 10 then 
            dprintf1 "    (after implied) tp = %s\n" (Layout.showL (typarDeclL tp));
         completeD)

    else
        let newConstraints = 
              (* Eliminate any constraints where one constraint implies another *)
              (* Keep constraints in the left-to-right form according to the order they are asserted. *)
              (* NOTE: QUADRATIC *)
              let rec eliminateRedundant cxs acc = 
                  match cxs with 
                  | [] ->  acc
                  | cx :: rest -> 
                      eliminateRedundant rest (if exists (fun cx2 -> implies cx2 cx) acc then acc else (cx::acc)) in 
                  
              eliminateRedundant allCxs [] in
              

        (* Write the constraint into the type variable *)
        (* Record a entry in the undo trace if one is provided *)
        let d = derefd_local_typar tp in 
        let orig = d.typar_constraints in 
        begin match trace with 
        | NoTrace -> () 
        | Trace actions -> actions := (fun () -> d.typar_constraints <- orig) :: !actions
        end;
        d.typar_constraints <- newConstraints;

        if verbose then dprintf1 "#newConstraints = %d\n" (length newConstraints);
        if verbose && length newConstraints > 10 then 
            dprintf1 "\n----------------------\n    tp = %s\n" (Layout.showL (typarDeclL tp));

        completeD)
  
and solveTypSupportsNull ndeep csenv m2 trace ty =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    begin 
        if is_typar_ty ty then recordConstraint ndeep csenv m2 trace (dest_typar_typ ty) (TTyparSupportsNull(m))
        else if typSatisfiesNullConstraint g ty then completeD
        else errorD (ConstraintSolverError(sprintf "The type '%s' does not have 'null' as a proper value" (NicePrint.pretty_string_of_typ denv ty),m,m2))
    end ++ (fun () -> 
    (* Require compile-time knowledge of the head type of all types used with the value 'null' *)
    (* REVIEW: this is almost certainly not needed. The 'null' constraint has to be discharged at some point *)
    (* and the only way we can do that is wih a head type. This it is a sufficeint description of the requirements on the variable *)
    solveTypStaticReq csenv trace HeadTypeStaticReq ty)

and solveTypIsEnum ndeep csenv m2 trace ty underlying =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    begin 
        if is_typar_ty ty then 
            recordConstraint ndeep csenv m2 trace (dest_typar_typ ty) (TTyparIsEnum(underlying,m))
        else if is_enum_typ ty then 
            solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace underlying (underlying_typ_of_enum_typ g ty) ++ (fun () ->
            completeD)
        else 
            errorD (ConstraintSolverError(sprintf "The type '%s' is not a .NET enum type" (NicePrint.pretty_string_of_typ denv ty),m,m2))
    end

and solveTypIsDelegate ndeep csenv m2 trace ty aty bty =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    begin 
        if is_typar_ty ty then 
            recordConstraint ndeep csenv m2 trace (dest_typar_typ ty) (TTyparIsDelegate(aty,bty,m))
        else if is_delegate_typ ty then 
            match try_dest_standard_delegate_ty g csenv.cs_css.css_amap m ty with 
            | Some (tupledArgTy,rty) ->
                solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace aty tupledArgTy ++ (fun () ->
                solveTypEqualsTypKeepAbbrevs ndeep csenv m2 trace bty rty ++ (fun () ->
                completeD))
            | None ->
                errorD (ConstraintSolverError(sprintf "The type '%s' has a non-standard delegate type" (NicePrint.pretty_string_of_typ denv ty),m,m2))
        else errorD (ConstraintSolverError(sprintf "The type '%s' is not a .NET delegate type" (NicePrint.pretty_string_of_typ denv ty),m,m2))
    end

and solveTypIsNonNullableValueType ndeep csenv m2 trace ty =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    if is_typar_ty ty then recordConstraint ndeep csenv m2 trace (dest_typar_typ ty) (TTyparIsNotNullableValueType(m))
    else if is_struct_typ ty then (
        (* IsValueType = IsValueType + NonNullable *)
        if g.typeCheckerConfiguredToAssumeV20Library && 
             not g.typeCheckerConfiguredToAssumeErasureOfGenerics && 
             g.tcref_eq g.system_Nullable_tcref (tcref_of_stripped_typ ty) then (

            errorD (ConstraintSolverError(sprintf "This type parameter may not be instantiated to 'Nullable'. This is a restriction imposed in order to ensure the meaning of 'null' in some .NET languages is not confusing when used in conjunction with 'Nullable' values",m,m))
        ) else (
            completeD
        )
    ) else (
        errorD (ConstraintSolverError(sprintf "The type '%s' is not a struct type" (NicePrint.pretty_string_of_typ denv ty),m,m2))
    )

and solveTypChoice ndeep csenv m2 trace ty tys =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    if is_typar_ty ty then recordConstraint ndeep csenv m2 trace (dest_typar_typ ty) (TTyparSimpleChoice(tys,m))
    else if List.exists (type_equiv g ty) tys then completeD
    else errorD (ConstraintSolverError(sprintf "The type '%s' is not compatible with any of the types %s, arising from the use of a printf-style format string" (NicePrint.pretty_string_of_typ denv ty) (String.concat "," (map (NicePrint.pretty_string_of_typ denv) tys)),m,m2))

and solveTypIsReferenceType ndeep csenv m2 trace ty =
    let g = csenv.cs_css.css_g in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    if is_typar_ty ty then recordConstraint ndeep csenv m2 trace (dest_typar_typ ty)  (TTyparIsReferenceType(m))
    else if is_ref_typ g ty then completeD
    else errorD (ConstraintSolverError(sprintf "The type '%s' is not a reference type" (NicePrint.pretty_string_of_typ denv ty),m,m))

and solveTypRequiresDefaultConstructor ndeep csenv m2 trace ty =
    let g = csenv.cs_css.css_g in 
    let amap = csenv.cs_css.css_amap in 
    let m = csenv.cs_m in 
    let denv = csenv.cs_denv in 
    if is_typar_ty ty then recordConstraint ndeep csenv m2 trace (dest_typar_typ ty) (TTyparRequiresDefaultConstructor(m))
    else if is_struct_typ ty then completeD
    else if
        intrinsic_cinfos_of_typ amap m ty 
        |> filter (minfo_accessible g amap m AccessibleFromEverywhere)   
        |> exists (minfo_is_nullary g)
           then completeD
    else errorD (ConstraintSolverError(sprintf "The type '%s' requires a public default constructor" (NicePrint.pretty_string_of_typ denv ty),m,m2))

(*-------------------------------------------------------------------------
!* Parameterized compatibility relation between member signatures.  The real work
 * is done by "check".
 *
 * QUERY: can we unify this relation with the member sig checks done up above?
 *------------------------------------------------------------------------- *)


and canMemberSigsMatchUpToCheck csenv permitOptArgs equateTypes subsumeArg reqdRetTyOpt 
      (CalledMeth(minfo,minst,uminst,unnamedCalledArgs,unnamedCallerArgs,methodRetTy,assignedNamedArgs,assignedNamedProps,pinfoOpt,unassignedNamedItem,attributeAssignedNamedItems,unnamedCalledOptArgs,unnamedCalledOutArgs)) : imperativeOperationResult =

    let g    = csenv.cs_css.css_g in
    let amap = csenv.cs_css.css_amap in
    let m    = csenv.cs_m in
    (* First equate the method instantiation (if any) with the method type parameters *)
    if length minst <> length uminst then errorD(Error("argument length mismatch",m)) else
    iter2D equateTypes minst uminst ++ (fun () -> 
    if not (permitOptArgs or isNil(unnamedCalledOptArgs)) then errorD(Error("optional arguments not permitted here",m)) else
    if length unnamedCalledArgs <> length unnamedCallerArgs then errorD(Error("argument length mismatch",m)) else
    (* Check all the argument types. *)
    iter2D subsumeArg unnamedCalledArgs unnamedCallerArgs ++ (fun () -> 
    iterD (fun (AssignedCalledArg(called,caller)) -> subsumeArg called caller) assignedNamedArgs ++ (fun () -> 
    iterD (fun (AssignedItemSetter(item,caller)) -> 
        let name, calledArgTy = 
            match item with
            | AssignedPropSetter(pminfo,pminst) -> 
                let calledArgTy = List.hd (argtys_of_minfo g amap m pminfo pminst) in
                name_of_minfo pminfo, calledArgTy

            | AssignedIlFieldSetter(finfo) ->
                (* Get or set instance IL field *)
                let calledArgTy = vtyp_of_il_finfo amap m  finfo in 
                name_of_il_finfo finfo, calledArgTy
            
            | AssignedRecdFieldSetter(rfinfo) ->
                let calledArgTy = vtyp_of_rfinfo rfinfo in 
                name_of_rfinfo rfinfo, calledArgTy in
        
        subsumeArg (CalledArg(-1,false, NotOptional,false,Some(name), calledArgTy)) caller) assignedNamedProps) ++ (fun () -> 
    
    (* If there is a conflict in the return type up to subsumption then reject the overload. *)
    (* This lets us use partial type information to resolve overloads such as op_Explicit *)
    (* Do not take into account return type information for constructors *)
    (* Take into account tupling up of unfilled out args *)
    if (minfo_is_ctor minfo) then completeD else
    match reqdRetTyOpt with 
       | None -> completeD 
       | Some (reqdRetTy,e) -> 
           let methodRetTy = 
               if isNil unnamedCalledOutArgs then methodRetTy else 
               let outArgTys = map (fun (CalledArg(i,_,_,_,_,argty)) -> dest_byref_ty g argty) unnamedCalledOutArgs in 
               if is_unit_typ g methodRetTy then mk_tupled_ty g outArgTys
               else mk_tupled_ty g (methodRetTy :: outArgTys) in
           subsumeArg (CalledArg(-1,false,NotOptional,false,None,reqdRetTy)) (CallerArg(methodRetTy,m,false,e))) ))

(*-------------------------------------------------------------------------
!* Resolve IL overloading. 
 * 
 * Note this utilizes the type inference constraint solving engine in undo mode.
 * 
 * REVIEW: move this to a separate file.
 *------------------------------------------------------------------------- *)



(* F# supports two adhoc conversions at method callsites (note C# supports more, though ones *)
(* such as implicit conversions interact badly with type inference). *)
(* The first is the use of "(fun x y -> ...)" when  a delegate it expected. This is not part of *)
(* the ":>" coercion relationship or inference constraint problem as *)
(* such, but is a special rule applied only to method arguments. *)
(* *)
(* The function adjustCalledArgType detects this case based on types and needs to know that the type being applied *)
(* is a function type. *)
(* *)
(* The other conversion supported is the two ways to pass a value where a byref is expxected. *)
(* The first (default) is to use a reference cell, and the interioer address is taken automatically *)
(* The second is an explicit use of the "address-of" operator "&e". Here we detect the second case, *)
(* and record the presence of the sytnax "&e" in the pre-inferred actual type for the method argument. *)
(* The function adjustCalledArgType detects this and refuses to apply the default byref-to-ref transformation. *)
(* *)
(* The function adjustCalledArgType also adjusts for optional arguments. *)
and adjustCalledArgType g amap (CalledArg(_,_,optArgInfo,isOutArg,_,calledArgTy)) (CallerArg(callerArgTy,m,isOptCallerArg,_)) =
    (* If the called method argument is a byref type, then the caller may provide a byref or ref *)
    if is_byref_ty g calledArgTy then
        if is_byref_ty g callerArgTy then 
            calledArgTy
        else 
            mk_refcell_ty g (dest_byref_ty g calledArgTy)  
    else 
        (* If the called method argument is a delegate type, then the caller may provide a function *)
        let calledArgTy = 
            if is_delegate_typ calledArgTy && is_fun_ty callerArgTy then 
                let minfo,del_argtys,del_rty,fty = sig_of_fun_as_delegate g amap calledArgTy m in
                let del_argtys = (if isNil del_argtys then [g.unit_ty] else del_argtys) in
                if length (fst (strip_fun_typ callerArgTy)) = length del_argtys
                then fty 
                else calledArgTy 
            else calledArgTy in

        (* Adjust the called argument type to take into account whether the caller's argument is M(?arg=Some(3)) or M(arg=1) *)
        (* If the called method argument is optional with type Option<T>, then the caller may provide a T, unless their argument is propogating-optional (i.e. isOptCallerArg) *)
        let calledArgTy = 
            match optArgInfo with 
            | NotOptional                    -> calledArgTy
            | CalleeSide when not isOptCallerArg && is_option_ty g calledArgTy  -> dest_option_ty g calledArgTy
            | CalleeSide | CallerSide _ -> calledArgTy in
        calledArgTy
        

and feasiblySubsumesOrConverts g amap calledArg (CallerArg(callerArgTy,m,_,_) as callerArg) =
    let calledArgTy = adjustCalledArgType g amap calledArg callerArg in
    if not (type_feasibly_subsumes_type 0 g amap m calledArgTy CanCoerce callerArgTy) then errorD(Error("The argument types don't match",m)) else
    completeD

and definitelyEquiv g amap calledArg (CallerArg(callerArgTy,m,_,_) as callerArg) = 
    let calledArgTy = adjustCalledArgType g amap calledArg callerArg in
    if not (type_equiv g calledArgTy callerArgTy) then errorD(Error("The argument types don't match",m)) else
    completeD
  
(* Assert a subtype cosntraint, and wrap an ErrorsFromAddingSubsumptionConstraint error around any failure *)
(* to allow us to report the outer types involved in the constraint *)
and solveTypSubsumesTypWithReport ndeep csenv m trace ty1 ty2 = 
    tryD (fun () -> solveTypSubsumesTyp ndeep csenv m trace ty1 ty2)
         (fun res -> errorD (ErrorsFromAddingSubsumptionConstraint(csenv.cs_css.css_g,csenv.cs_denv,ty1,ty2,res,m)))

and solveTypEqualsTypWithReport ndeep csenv m trace ty1 ty2 = 
    tryD (fun () -> solveTypEqualsTypKeepAbbrevs ndeep csenv m trace ty1 ty2)
         (fun res -> errorD (ErrorFromAddingTypeEquation(csenv.cs_css.css_g,csenv.cs_denv,ty1,ty2,res,m)))
  
and mustSubsumeOrConvert 
        csenv trace
        (CalledArg(_,isParamArrayArg,_,_,_,calledArgTy) as calledArg) 
        (CallerArg(callerArgTy,m,_,_) as callerArg) = 
        
    let g = csenv.cs_css.css_g in
    let amap = csenv.cs_css.css_amap in
    let calledArgTy = adjustCalledArgType g amap calledArg callerArg in
    solveTypSubsumesTypWithReport 0 csenv m trace calledArgTy callerArgTy ++ (fun () -> 

    if isParamArrayArg &&
        is_stripped_tyapp_typ calledArgTy &&
        (let tcf,tinstf = dest_stripped_tyapp_typ calledArgTy in
        List.length tinstf  = 1 &&
        type_feasibly_equiv 0 g amap m (List.hd tinstf) callerArgTy)
    then 
        errorD(Error("This method expects a .NET 'params' parameter in this position. 'params' is a way of passing a variable number of arguments to a method in languages such as C#. Consider passing an array for this argument",m))
    else
        completeD)

and mustUnify csenv trace ty1 ty2 = 
    let m    = csenv.cs_m in
    solveTypEqualsTypWithReport 0 csenv m trace ty1 ty2

and mustUnifyInsideUndo csenv trace ty1 ty2 = 
    solveTypEqualsTypWithReport 0 csenv csenv.cs_m (Trace trace) ty1 ty2

and mustSubsumeOrConvertInsideUndo csenv trace calledArg (CallerArg(callerArgTy,m,_,_) as callerArg) = 
    let g = csenv.cs_css.css_g in
    let amap = csenv.cs_css.css_amap in
    let calledArgTy = adjustCalledArgType g amap calledArg callerArg in
    solveTypSubsumesTypWithReport 0 csenv m (Trace trace) calledArgTy callerArgTy 

and reportNoCandidatesError csenv (nUnnamedCallerArgs,nNamedCallerArgs) methodName ad calledMethGroup =

    let g    = csenv.cs_css.css_g in
    let amap = csenv.cs_css.css_amap in
    let m    = csenv.cs_m in
    let denv = csenv.cs_denv in
    if verbose then  dprintf2 "reportNoCandidatesError@%a\n" output_range m;
    match partition (minfo_of_cmeth >> minfo_accessible g amap m ad) calledMethGroup,
          partition cmethCorrectArity calledMethGroup,
          partition cmethAssignsAllNamedArgs calledMethGroup with
    (* No version accessible *)
    | ([],others),_,_ ->  
        errorD (Error ("The member or object constructor '"^methodName^"' is not "^showAccessDomain ad ^
                       (if nonNil others then ". Private members may only be accessed from within the declaring type. Protected members may only be accessed from an extending type and may not be accessed from inner lambda expressions" else ""), m))
    (* One method, incorrect name/arg assignment *)
    | _,_,([],[cmeth]) -> 
        let msg = List.fold_right 
             (fun (CallerNamedArg(nm,CallerArg(_,m,_,_))) acc -> "The member or object constructor '"^methodName^"' has no argument or settable return property '"^nm^"'. "^acc)
             (unassigned_named_args_of_cmeth cmeth )
             ("The required signature is "^string_of_minfo g amap m denv (minfo_of_cmeth cmeth )) in
        errorD (Error (msg,m))
    (* One method, incorrect arity *)
    | _,([],[cmeth]),_ when not (cmethCorrectArity cmeth) ->  
        let minfo = minfo_of_cmeth cmeth in 
        let nReqd = numUnnamedCalledArgs_of_cmeth cmeth in
        let nReqdNamed = numAssignedArgs_of_cmeth cmeth in
        let nActual = numUnnamedCallerArgs_of_cmeth cmeth in 
        let nreqdTyArgs = numCalledTyArgs_of_cmeth cmeth in
        let nactualTyArgs = numCallerTyArgs_of_cmeth cmeth in 
        if nActual <> nReqd then 
            if numAssignedArgs_of_cmeth cmeth > 0 or numAssignedProps_of_cmeth cmeth > 0 then 
                let nameText = 
                    if nReqd > nActual then 
                        let missingArgs = drop nReqd (unnamedCalledArgs_of_cmeth cmeth) in 
                        let names = namesOfCalledArgs missingArgs in 
                        if names = [] then "" else
                        ". Some names for missing arguments are "^String.concat ";" names
                    else "" in
                let furtherText = if nActual = 0 then "" else " additional" in
                if nReqd > nActual then 
                    errorD (Error ("The member or object constructor '"^methodName^"' requires "^string_of_int (nReqd-nActual)^furtherText^" argument(s). The required signature is '"^string_of_minfo g amap m denv minfo^"'"^nameText, m))
                else 
                    errorD (Error ("The member or object constructor '"^methodName^"' requires "^string_of_int (nReqd+nReqdNamed)^" argument(s) but is here given "^string_of_int nActual^" unnamed and "^(string_of_int nReqdNamed)^" named argument(s). The required signature is '"^string_of_minfo g amap m denv minfo^"'", m))
            else
                errorD (Error ("The member or object constructor '"^methodName^"' takes "^string_of_int nReqd^" argument(s) but is here given "^string_of_int nActual^". The required signature is '"^string_of_minfo g amap m denv minfo^"'", m))
        else 
            errorD (Error ("The member or object constructor '"^methodName^"' takes "^string_of_int nreqdTyArgs^" type argument(s) but is here given "^string_of_int nactualTyArgs^". The required signature is '"^string_of_minfo g amap m denv minfo^"'", m))

    (* One or more accessible, all the same arity, none correct *)
    | ((cmeth :: minfos),_),_,_ when not (cmethCorrectArity cmeth) && List.for_all (fun cmeth2 -> numUnnamedCalledArgs_of_cmeth cmeth = numUnnamedCalledArgs_of_cmeth cmeth2) minfos -> 
        errorD (Error ("The member or object constructor '"^methodName^"' taking "^string_of_int (numUnnamedCallerArgs_of_cmeth cmeth)^" arguments is not accessible from this code location. All accessible versions of method '"^methodName^"' take "^string_of_int (numUnnamedCalledArgs_of_cmeth cmeth)^" arguments",m))
    (* Many methods of different arities, all incorrect *)
    | _,([],(cmeth :: _)),_ -> 
        let minfo = minfo_of_cmeth cmeth in 
        if cmethCorrectArity cmeth then 
            errorD (Error ("No "^showAccessDomain ad^" member or object constructor named '"^methodName^"' matches the arguments. A possible signature is "^string_of_minfo g amap m denv minfo, m))
        else  
            errorD (Error ("The member or object constructor '"^methodName^"' does not take "^string_of_int (numUnnamedCallerArgs_of_cmeth cmeth)^" argument(s). An overload was found taking "^string_of_int (arity_of_minfo g minfo)^" arguments",m))
    | _ -> 
        let msg = "No "^showAccessDomain ad^" member or object constructor named '"^methodName^"' takes "^string_of_int nUnnamedCallerArgs^" arguments" in 
        let msg = if nNamedCallerArgs = 0 then msg else msg^". Note the call to this member also provides "^string_of_int nUnnamedCallerArgs^" named arguments" in
        errorD (Error (msg,m))


(* Resolve the overloading of a method *)
(* This is used after analyzing the types of arguments *)
and resolveOverloading 
         csenv 
         trace
         methodName      (* The name of the method being called, for error reporting *)
         callerArgCounts (* How many named/unnamed args id the caller provide? *)
         ad              (* The access domain of the caller, e.g. a module, type etc. *)
         calledMethGroup (* The set of methods being called *)
         reqdRetTyOpt          (* The expected return type, if known *)
     =
    let g = csenv.cs_css.css_g in
    let amap = csenv.cs_css.css_amap in
    let m    = csenv.cs_m in
    let denv = csenv.cs_denv in
    (* See what candidates we have based on name and arity *)
    let candidates = filter (cmethIsCandidate g amap m ad) calledMethGroup in
    let calledMethOpt, errors = 

        match calledMethGroup,candidates with 
        | _,[calledMeth] -> 
            Some(calledMeth), completeD

        | [],_ -> 
            None, errorD (Error (sprintf "Method or object constructor '%s' not found" methodName,m))

        | _,[] -> 
            None, reportNoCandidatesError csenv callerArgCounts methodName ad calledMethGroup
            
        | _,_ -> 
            (* See what candidates we have based on current inferred type information and exact matches of argument types *)
            (* Return type deliberately not take into account *)
            match filterEachWithUndo (fun newTrace calledMeth -> canMemberSigsMatchUpToCheck csenv true (mustUnifyInsideUndo csenv newTrace) (definitelyEquiv g amap) None calledMeth) candidates with
            | [calledMeth] -> 
                Some(calledMeth), completeD

            | _ -> 
                (* See what candidates we have based on current inferred type information and subsumption, *)
                (* including of return types *)
                match filterEachWithUndo (fun newTrace calledMeth -> canMemberSigsMatchUpToCheck csenv true (mustUnifyInsideUndo csenv newTrace) (feasiblySubsumesOrConverts g amap) reqdRetTyOpt calledMeth) candidates with
                | [] -> 
                    let errors = 
                        (candidates |> chooseList (fun calledMeth -> 
                                match collectThenUndo (fun newTrace -> canMemberSigsMatchUpToCheck csenv true (mustUnifyInsideUndo csenv newTrace) (mustSubsumeOrConvertInsideUndo csenv newTrace) reqdRetTyOpt calledMeth) with 
                                | OkResult _ -> None
                                | ErrorResult(_,exn) -> Some exn)) in
                                                    
                    let overloads = possibleOverloads g amap m denv calledMethGroup in

                    None,errorD (UnresolvedOverloading (denv,overloads,errors,"No overloads match for method "^methodName^". Possible matches are shown below (or in the Error List window)",m))

                | [calledMeth] -> 
                    Some(calledMeth), completeD

                | candidates -> 
                    match filterEachWithUndo (fun newTrace calledMeth -> canMemberSigsMatchUpToCheck csenv true (mustUnifyInsideUndo csenv newTrace) (mustSubsumeOrConvertInsideUndo csenv newTrace) reqdRetTyOpt calledMeth) candidates with
                    | [calledMeth] -> 
                        Some(calledMeth), completeD

                    | [] -> 
                        None, errorD (Error ("No overloads match for method "^methodName,m))
                        
                    | calledMethGroup -> 
                        (* OK, more than one solution to the constraint puzzle *)
                        (* Check if there is a unique "best" one *)
                        (* First look if there is one without any byref out args being returned via a tuple *)
                        match calledMethGroup |> filter (cmeth_has_out_args >> not) with
                        | [calledMeth] -> Some(calledMeth), completeD
                        | _ -> 
                            let overloads = possibleOverloads  g amap m denv calledMethGroup in
                            None, 
                            errorD (UnresolvedOverloading (denv,overloads,[],"The method '"^methodName^"' is overloaded. Possible matches are shown below (or in the Error List window)",m)); in
                            
    (* If we've got a candidate solution: make the final checks - no undo here! *)
    match calledMethOpt with 
    | Some(calledMeth) -> 
        calledMethOpt,
        errors ++ (fun () -> canMemberSigsMatchUpToCheck csenv true (mustUnify csenv trace) (mustSubsumeOrConvert csenv trace) reqdRetTyOpt calledMeth)

    | None -> 
        None, errors        


(* This is used before analyzing the types of arguments *)
let unifyUniqueOverloading csenv callerArgCounts methodName ad calledMethGroup =
    let g = csenv.cs_css.css_g in
    let amap = csenv.cs_css.css_amap in
    let m    = csenv.cs_m in
    let denv = csenv.cs_denv in
    if verbose then  dprintf2 "--> unifyUniqueOverloading@%a\n" output_range m;
    (* See what candidates we have based on name and arity *)
    let candidates = filter (cmethIsCandidate g amap m ad) calledMethGroup in
    if verbose then  dprintf2 "in unifyUniqueOverloading@%a\n" output_range m;
    match calledMethGroup,candidates with 
    | _,[calledMeth] -> 
        (* Only one candidate found - we thus know the types we expect of arguments *)
        canMemberSigsMatchUpToCheck csenv true (mustUnify csenv NoTrace) (mustSubsumeOrConvert csenv NoTrace) None calledMeth
        ++ (fun () -> resultD(true))
        
    | [],_ -> 
        errorD (Error ("Method or object constructor '"^methodName^"' not found",m))
    | _,[] -> 
        reportNoCandidatesError csenv callerArgCounts methodName ad calledMethGroup 
        ++ (fun () -> resultD(false))
    | _ -> 
        resultD(false)




let eliminateConstraintsForGeneralizedTypars csenv trace generalizedTypars =
    (* Resolve the global constraints where this type variable appears in the support of the constraint *)
    generalizedTypars |> List.iter (fun tp -> 
        let tpn = (stamp_of_typar tp) in
        let cxst = csenv.cs_css.css_cxs in 
        let cxs = Hashtbl.find_all  cxst tpn in
        if isNil cxs then () else
        if verbose then dprintf3 "eliminateConstraintsForGeneralizedTypars: #cxs = %d, m = %a\n" (List.length cxs) output_range csenv.cs_m;
        cxs |> List.iter (fun cx -> 
            Hashtbl.remove cxst tpn;
            match trace with 
            | NoTrace -> () 
            | Trace actions -> actions := (fun () -> (Hashtbl.add csenv.cs_css.css_cxs tpn cx)) :: !actions)
    )


(*-------------------------------------------------------------------------
!* Main entry points to constraint solver (some backdoors are used for 
 * some constructs)
 *
 * No error recovery here : we do that on a per-expression basis.
 *------------------------------------------------------------------------- *)

let unify denv css m ty1 ty2 = 
    solveTypEqualsTypWithReport 0 (mk_csenv css m denv) m NoTrace ty1 ty2
    |> raiseOperationResult

let undo_if_failed f =
    let trace = newTrace() in 
    let res = 
        try f trace |> gaveNoErrors
        with e -> false in 
    if not res then undoTrace trace;
    res

let unify_and_undo_if_failed denv css m ty1 ty2 =
    undo_if_failed (fun trace -> solveTypEqualsTypKeepAbbrevs 0 (mk_csenv css m denv) m (Trace(trace)) ty1 ty2)

let type_must_subsume_type_and_undo_if_failed denv css m ty1 ty2 = 
    undo_if_failed (fun trace -> solveTypSubsumesTyp 0 (mk_csenv css m denv) m (Trace(trace)) ty1 ty2)

let type_must_subsume_type denv css m trace ty1 ty2 = 
    solveTypSubsumesTypWithReport 0 (mk_csenv css m denv) m trace ty1 ty2
    |> raiseOperationResult

let type_must_support_trait denv css m trace traitInfo  =
    tryD (fun () -> solveMemberConstraint false 0 (mk_csenv css m denv) m trace traitInfo ++ (fun _ -> completeD))
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m)))
    |> raiseOperationResult

let type_must_support_null denv css m trace ty =
    tryD (fun () -> solveTypSupportsNull 0 (mk_csenv css m denv) m trace ty)
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m)))
    |> raiseOperationResult

let type_must_support_default_ctor denv css m trace ty =
    tryD (fun () -> solveTypRequiresDefaultConstructor 0 (mk_csenv css m denv) m trace ty)
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m)))
    |> raiseOperationResult

let type_is_reference_type denv css m trace ty =
    tryD (fun () -> solveTypIsReferenceType 0  (mk_csenv css m denv) m trace ty)
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m)))
    |> raiseOperationResult

let type_is_value_type denv css m trace ty =
    tryD (fun () -> solveTypIsNonNullableValueType 0 (mk_csenv css m denv) m trace ty)
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m)))
    |> raiseOperationResult

let type_is_enum denv css m trace ty underlying =
    tryD (fun () -> solveTypIsEnum 0 (mk_csenv css m denv) m trace ty underlying)
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m)))
    |> raiseOperationResult

let type_is_delegate denv css m trace ty aty bty =
    tryD (fun () -> solveTypIsDelegate 0 (mk_csenv css m denv) m trace ty aty bty)
         (fun res -> errorD (ErrorFromAddingConstraint(denv,res,m)))
    |> raiseOperationResult


let codegen_witnessThatTypSupportsTraitConstraint g amap m traitInfo = 
    let css = {css_g=g;css_amap=amap;css_cxs=Hashtbl.create 10 } in
    let csenv = mk_csenv css m (empty_denv g) in 
    solveMemberConstraint true 0 csenv m NoTrace traitInfo ++ (fun soln -> 
       match soln with 
       | TTraitUnsolved -> errorD(InternalError("unsolved trait constraint in codegen",m))
       | TTraitBuiltIn -> errorD(InternalError("trait constraint was resolved to F# library intrinsic in codegen",m))
       | TTraitSolved minfo -> resultD minfo)

    

let choose_typar_solution_and_solve css denv tp =
    let g = css.css_g in
    let amap = css.css_amap in
    let max,m = choose_typar_solution_and_range g amap tp  in
    let csenv = (mk_csenv css m denv) in 
    tryD (fun () -> solveTyparEqualsTyp 0 csenv m NoTrace (mk_typar_ty tp) max)
         (fun err -> errorD(ErrorFromApplyingDefault(g,denv,tp,max,err,m)))
    |> raiseOperationResult


