---@meta

---
---Provides an interface to the user's keyboard.
---
---@class love.keyboard
love.keyboard = {}

---
---Gets the key corresponding to the given hardware scancode.
---
---Unlike key constants, Scancodes are keyboard layout-independent. For example the scancode 'w' will be generated if the key in the same place as the 'w' key on an American keyboard is pressed, no matter what the key is labelled or what the user's operating system settings are.
---
---Scancodes are useful for creating default controls that have the same physical locations on on all systems.
---
---@param scancode love.Scancode # The scancode to get the key from.
---@return love.KeyConstant key # The key corresponding to the given scancode, or 'unknown' if the scancode doesn't map to a KeyConstant on the current system.
function love.keyboard.getKeyFromScancode(scancode) end

---
---Gets the hardware scancode corresponding to the given key.
---
---Unlike key constants, Scancodes are keyboard layout-independent. For example the scancode 'w' will be generated if the key in the same place as the 'w' key on an American keyboard is pressed, no matter what the key is labelled or what the user's operating system settings are.
---
---Scancodes are useful for creating default controls that have the same physical locations on on all systems.
---
---@param key love.KeyConstant # The key to get the scancode from.
---@return love.Scancode scancode # The scancode corresponding to the given key, or 'unknown' if the given key has no known physical representation on the current system.
function love.keyboard.getScancodeFromKey(key) end

---
---Gets whether key repeat is enabled.
---
---@return boolean enabled # Whether key repeat is enabled.
function love.keyboard.hasKeyRepeat() end

---
---Gets whether screen keyboard is supported.
---
---@return boolean supported # Whether screen keyboard is supported.
function love.keyboard.hasScreenKeyboard() end

---
---Gets whether text input events are enabled.
---
---@return boolean enabled # Whether text input events are enabled.
function love.keyboard.hasTextInput() end

---
---Checks whether a certain key is down. Not to be confused with love.keypressed or love.keyreleased.
---
---@overload fun(key: love.KeyConstant, ...):boolean
---@param key love.KeyConstant # The key to check.
---@return boolean down # True if the key is down, false if not.
function love.keyboard.isDown(key) end

---
---Checks whether the specified Scancodes are pressed. Not to be confused with love.keypressed or love.keyreleased.
---
---Unlike regular KeyConstants, Scancodes are keyboard layout-independent. The scancode 'w' is used if the key in the same place as the 'w' key on an American keyboard is pressed, no matter what the key is labelled or what the user's operating system settings are.
---
---@param scancode love.Scancode # A Scancode to check.
---@vararg love.Scancode # Additional Scancodes to check.
---@return boolean down # True if any supplied Scancode is down, false if not.
function love.keyboard.isScancodeDown(scancode, ...) end

---
---Enables or disables key repeat for love.keypressed. It is disabled by default.
---
---@param enable boolean # Whether repeat keypress events should be enabled when a key is held down.
function love.keyboard.setKeyRepeat(enable) end

---
---Enables or disables text input events. It is enabled by default on Windows, Mac, and Linux, and disabled by default on iOS and Android.
---
---On touch devices, this shows the system's native on-screen keyboard when it's enabled.
---
---@overload fun(enable: boolean, x: number, y: number, w: number, h: number)
---@param enable boolean # Whether text input events should be enabled.
function love.keyboard.setTextInput(enable) end

---
---All the keys you can press. Note that some keys may not be available on your keyboard or system.
---
---@alias love.KeyConstant
---
---The A key
---
---| "a"
---
---The B key
---
---| "b"
---
---The C key
---
---| "c"
---
---The D key
---
---| "d"
---
---The E key
---
---| "e"
---
---The F key
---
---| "f"
---
---The G key
---
---| "g"
---
---The H key
---
---| "h"
---
---The I key
---
---| "i"
---
---The J key
---
---| "j"
---
---The K key
---
---| "k"
---
---The L key
---
---| "l"
---
---The M key
---
---| "m"
---
---The N key
---
---| "n"
---
---The O key
---
---| "o"
---
---The P key
---
---| "p"
---
---The Q key
---
---| "q"
---
---The R key
---
---| "r"
---
---The S key
---
---| "s"
---
---The T key
---
---| "t"
---
---The U key
---
---| "u"
---
---The V key
---
---| "v"
---
---The W key
---
---| "w"
---
---The X key
---
---| "x"
---
---The Y key
---
---| "y"
---
---The Z key
---
---| "z"
---
---The zero key
---
---| "0"
---
---The one key
---
---| "1"
---
---The two key
---
---| "2"
---
---The three key
---
---| "3"
---
---The four key
---
---| "4"
---
---The five key
---
---| "5"
---
---The six key
---
---| "6"
---
---The seven key
---
---| "7"
---
---The eight key
---
---| "8"
---
---The nine key
---
---| "9"
---
---Space key
---
---| "space"
---
---Exclamation mark key
---
---| "!"
---
---Double quote key
---
---| "\""
---
---Hash key
---
---| "#"
---
---Dollar key
---
---| "$"
---
---Ampersand key
---
---| "&"
---
---Single quote key
---
---| "'"
---
---Left parenthesis key
---
---| "("
---
---Right parenthesis key
---
---| ")"
---
---Asterisk key
---
---| "*"
---
---Plus key
---
---| "+"
---
---Comma key
---
---| ","
---
---Hyphen-minus key
---
---| "-"
---
---Full stop key
---
---| "."
---
---Slash key
---
---| "/"
---
---Colon key
---
---| ":"
---
---Semicolon key
---
---| ";"
---
---Less-than key
---
---| "<"
---
---Equal key
---
---| "="
---
---Greater-than key
---
---| ">"
---
---Question mark key
---
---| "?"
---
---At sign key
---
---| "@"
---
---Left square bracket key
---
---| "["
---
---Backslash key
---
---| "\\"
---
---Right square bracket key
---
---| "]"
---
---Caret key
---
---| "^"
---
---Underscore key
---
---| "_"
---
---Grave accent key
---
---| "`"
---
---The numpad zero key
---
---| "kp0"
---
---The numpad one key
---
---| "kp1"
---
---The numpad two key
---
---| "kp2"
---
---The numpad three key
---
---| "kp3"
---
---The numpad four key
---
---| "kp4"
---
---The numpad five key
---
---| "kp5"
---
---The numpad six key
---
---| "kp6"
---
---The numpad seven key
---
---| "kp7"
---
---The numpad eight key
---
---| "kp8"
---
---The numpad nine key
---
---| "kp9"
---
---The numpad decimal point key
---
---| "kp."
---
---The numpad division key
---
---| "kp/"
---
---The numpad multiplication key
---
---| "kp*"
---
---The numpad substraction key
---
---| "kp-"
---
---The numpad addition key
---
---| "kp+"
---
---The numpad enter key
---
---| "kpenter"
---
---The numpad equals key
---
---| "kp="
---
---Up cursor key
---
---| "up"
---
---Down cursor key
---
---| "down"
---
---Right cursor key
---
---| "right"
---
---Left cursor key
---
---| "left"
---
---Home key
---
---| "home"
---
---End key
---
---| "end"
---
---Page up key
---
---| "pageup"
---
---Page down key
---
---| "pagedown"
---
---Insert key
---
---| "insert"
---
---Backspace key
---
---| "backspace"
---
---Tab key
---
---| "tab"
---
---Clear key
---
---| "clear"
---
---Return key
---
---| "return"
---
---Delete key
---
---| "delete"
---
---The 1st function key
---
---| "f1"
---
---The 2nd function key
---
---| "f2"
---
---The 3rd function key
---
---| "f3"
---
---The 4th function key
---
---| "f4"
---
---The 5th function key
---
---| "f5"
---
---The 6th function key
---
---| "f6"
---
---The 7th function key
---
---| "f7"
---
---The 8th function key
---
---| "f8"
---
---The 9th function key
---
---| "f9"
---
---The 10th function key
---
---| "f10"
---
---The 11th function key
---
---| "f11"
---
---The 12th function key
---
---| "f12"
---
---The 13th function key
---
---| "f13"
---
---The 14th function key
---
---| "f14"
---
---The 15th function key
---
---| "f15"
---
---Num-lock key
---
---| "numlock"
---
---Caps-lock key
---
---| "capslock"
---
---Scroll-lock key
---
---| "scrollock"
---
---Right shift key
---
---| "rshift"
---
---Left shift key
---
---| "lshift"
---
---Right control key
---
---| "rctrl"
---
---Left control key
---
---| "lctrl"
---
---Right alt key
---
---| "ralt"
---
---Left alt key
---
---| "lalt"
---
---Right meta key
---
---| "rmeta"
---
---Left meta key
---
---| "lmeta"
---
---Left super key
---
---| "lsuper"
---
---Right super key
---
---| "rsuper"
---
---Mode key
---
---| "mode"
---
---Compose key
---
---| "compose"
---
---Pause key
---
---| "pause"
---
---Escape key
---
---| "escape"
---
---Help key
---
---| "help"
---
---Print key
---
---| "print"
---
---System request key
---
---| "sysreq"
---
---Break key
---
---| "break"
---
---Menu key
---
---| "menu"
---
---Power key
---
---| "power"
---
---Euro (&euro;) key
---
---| "euro"
---
---Undo key
---
---| "undo"
---
---WWW key
---
---| "www"
---
---Mail key
---
---| "mail"
---
---Calculator key
---
---| "calculator"
---
---Application search key
---
---| "appsearch"
---
---Application home key
---
---| "apphome"
---
---Application back key
---
---| "appback"
---
---Application forward key
---
---| "appforward"
---
---Application refresh key
---
---| "apprefresh"
---
---Application bookmarks key
---
---| "appbookmarks"

---
---Keyboard scancodes.
---
---Scancodes are keyboard layout-independent, so the scancode "w" will be generated if the key in the same place as the "w" key on an American QWERTY keyboard is pressed, no matter what the key is labelled or what the user's operating system settings are.
---
---Using scancodes, rather than keycodes, is useful because keyboards with layouts differing from the US/UK layout(s) might have keys that generate 'unknown' keycodes, but the scancodes will still be detected. This however would necessitate having a list for each keyboard layout one would choose to support.
---
---One could use textinput or textedited instead, but those only give back the end result of keys used, i.e. you can't get modifiers on their own from it, only the final symbols that were generated.
---
---@alias love.Scancode
---
---The 'A' key on an American layout.
---
---| "a"
---
---The 'B' key on an American layout.
---
---| "b"
---
---The 'C' key on an American layout.
---
---| "c"
---
---The 'D' key on an American layout.
---
---| "d"
---
---The 'E' key on an American layout.
---
---| "e"
---
---The 'F' key on an American layout.
---
---| "f"
---
---The 'G' key on an American layout.
---
---| "g"
---
---The 'H' key on an American layout.
---
---| "h"
---
---The 'I' key on an American layout.
---
---| "i"
---
---The 'J' key on an American layout.
---
---| "j"
---
---The 'K' key on an American layout.
---
---| "k"
---
---The 'L' key on an American layout.
---
---| "l"
---
---The 'M' key on an American layout.
---
---| "m"
---
---The 'N' key on an American layout.
---
---| "n"
---
---The 'O' key on an American layout.
---
---| "o"
---
---The 'P' key on an American layout.
---
---| "p"
---
---The 'Q' key on an American layout.
---
---| "q"
---
---The 'R' key on an American layout.
---
---| "r"
---
---The 'S' key on an American layout.
---
---| "s"
---
---The 'T' key on an American layout.
---
---| "t"
---
---The 'U' key on an American layout.
---
---| "u"
---
---The 'V' key on an American layout.
---
---| "v"
---
---The 'W' key on an American layout.
---
---| "w"
---
---The 'X' key on an American layout.
---
---| "x"
---
---The 'Y' key on an American layout.
---
---| "y"
---
---The 'Z' key on an American layout.
---
---| "z"
---
---The '1' key on an American layout.
---
---| "1"
---
---The '2' key on an American layout.
---
---| "2"
---
---The '3' key on an American layout.
---
---| "3"
---
---The '4' key on an American layout.
---
---| "4"
---
---The '5' key on an American layout.
---
---| "5"
---
---The '6' key on an American layout.
---
---| "6"
---
---The '7' key on an American layout.
---
---| "7"
---
---The '8' key on an American layout.
---
---| "8"
---
---The '9' key on an American layout.
---
---| "9"
---
---The '0' key on an American layout.
---
---| "0"
---
---The 'return' / 'enter' key on an American layout.
---
---| "return"
---
---The 'escape' key on an American layout.
---
---| "escape"
---
---The 'backspace' key on an American layout.
---
---| "backspace"
---
---The 'tab' key on an American layout.
---
---| "tab"
---
---The spacebar on an American layout.
---
---| "space"
---
---The minus key on an American layout.
---
---| "-"
---
---The equals key on an American layout.
---
---| "="
---
---The left-bracket key on an American layout.
---
---| "["
---
---The right-bracket key on an American layout.
---
---| "]"
---
---The backslash key on an American layout.
---
---| "\\"
---
---The non-U.S. hash scancode.
---
---| "nonus#"
---
---The semicolon key on an American layout.
---
---| ";"
---
---The apostrophe key on an American layout.
---
---| "'"
---
---The back-tick / grave key on an American layout.
---
---| "`"
---
---The comma key on an American layout.
---
---| ","
---
---The period key on an American layout.
---
---| "."
---
---The forward-slash key on an American layout.
---
---| "/"
---
---The capslock key on an American layout.
---
---| "capslock"
---
---The F1 key on an American layout.
---
---| "f1"
---
---The F2 key on an American layout.
---
---| "f2"
---
---The F3 key on an American layout.
---
---| "f3"
---
---The F4 key on an American layout.
---
---| "f4"
---
---The F5 key on an American layout.
---
---| "f5"
---
---The F6 key on an American layout.
---
---| "f6"
---
---The F7 key on an American layout.
---
---| "f7"
---
---The F8 key on an American layout.
---
---| "f8"
---
---The F9 key on an American layout.
---
---| "f9"
---
---The F10 key on an American layout.
---
---| "f10"
---
---The F11 key on an American layout.
---
---| "f11"
---
---The F12 key on an American layout.
---
---| "f12"
---
---The F13 key on an American layout.
---
---| "f13"
---
---The F14 key on an American layout.
---
---| "f14"
---
---The F15 key on an American layout.
---
---| "f15"
---
---The F16 key on an American layout.
---
---| "f16"
---
---The F17 key on an American layout.
---
---| "f17"
---
---The F18 key on an American layout.
---
---| "f18"
---
---The F19 key on an American layout.
---
---| "f19"
---
---The F20 key on an American layout.
---
---| "f20"
---
---The F21 key on an American layout.
---
---| "f21"
---
---The F22 key on an American layout.
---
---| "f22"
---
---The F23 key on an American layout.
---
---| "f23"
---
---The F24 key on an American layout.
---
---| "f24"
---
---The left control key on an American layout.
---
---| "lctrl"
---
---The left shift key on an American layout.
---
---| "lshift"
---
---The left alt / option key on an American layout.
---
---| "lalt"
---
---The left GUI (command / windows / super) key on an American layout.
---
---| "lgui"
---
---The right control key on an American layout.
---
---| "rctrl"
---
---The right shift key on an American layout.
---
---| "rshift"
---
---The right alt / option key on an American layout.
---
---| "ralt"
---
---The right GUI (command / windows / super) key on an American layout.
---
---| "rgui"
---
---The printscreen key on an American layout.
---
---| "printscreen"
---
---The scroll-lock key on an American layout.
---
---| "scrolllock"
---
---The pause key on an American layout.
---
---| "pause"
---
---The insert key on an American layout.
---
---| "insert"
---
---The home key on an American layout.
---
---| "home"
---
---The numlock / clear key on an American layout.
---
---| "numlock"
---
---The page-up key on an American layout.
---
---| "pageup"
---
---The forward-delete key on an American layout.
---
---| "delete"
---
---The end key on an American layout.
---
---| "end"
---
---The page-down key on an American layout.
---
---| "pagedown"
---
---The right-arrow key on an American layout.
---
---| "right"
---
---The left-arrow key on an American layout.
---
---| "left"
---
---The down-arrow key on an American layout.
---
---| "down"
---
---The up-arrow key on an American layout.
---
---| "up"
---
---The non-U.S. backslash scancode.
---
---| "nonusbackslash"
---
---The application key on an American layout. Windows contextual menu, compose key.
---
---| "application"
---
---The 'execute' key on an American layout.
---
---| "execute"
---
---The 'help' key on an American layout.
---
---| "help"
---
---The 'menu' key on an American layout.
---
---| "menu"
---
---The 'select' key on an American layout.
---
---| "select"
---
---The 'stop' key on an American layout.
---
---| "stop"
---
---The 'again' key on an American layout.
---
---| "again"
---
---The 'undo' key on an American layout.
---
---| "undo"
---
---The 'cut' key on an American layout.
---
---| "cut"
---
---The 'copy' key on an American layout.
---
---| "copy"
---
---The 'paste' key on an American layout.
---
---| "paste"
---
---The 'find' key on an American layout.
---
---| "find"
---
---The keypad forward-slash key on an American layout.
---
---| "kp/"
---
---The keypad '*' key on an American layout.
---
---| "kp*"
---
---The keypad minus key on an American layout.
---
---| "kp-"
---
---The keypad plus key on an American layout.
---
---| "kp+"
---
---The keypad equals key on an American layout.
---
---| "kp="
---
---The keypad enter key on an American layout.
---
---| "kpenter"
---
---The keypad '1' key on an American layout.
---
---| "kp1"
---
---The keypad '2' key on an American layout.
---
---| "kp2"
---
---The keypad '3' key on an American layout.
---
---| "kp3"
---
---The keypad '4' key on an American layout.
---
---| "kp4"
---
---The keypad '5' key on an American layout.
---
---| "kp5"
---
---The keypad '6' key on an American layout.
---
---| "kp6"
---
---The keypad '7' key on an American layout.
---
---| "kp7"
---
---The keypad '8' key on an American layout.
---
---| "kp8"
---
---The keypad '9' key on an American layout.
---
---| "kp9"
---
---The keypad '0' key on an American layout.
---
---| "kp0"
---
---The keypad period key on an American layout.
---
---| "kp."
---
---The 1st international key on an American layout. Used on Asian keyboards.
---
---| "international1"
---
---The 2nd international key on an American layout.
---
---| "international2"
---
---The 3rd international  key on an American layout. Yen.
---
---| "international3"
---
---The 4th international key on an American layout.
---
---| "international4"
---
---The 5th international key on an American layout.
---
---| "international5"
---
---The 6th international key on an American layout.
---
---| "international6"
---
---The 7th international key on an American layout.
---
---| "international7"
---
---The 8th international key on an American layout.
---
---| "international8"
---
---The 9th international key on an American layout.
---
---| "international9"
---
---Hangul/English toggle scancode.
---
---| "lang1"
---
---Hanja conversion scancode.
---
---| "lang2"
---
---Katakana scancode.
---
---| "lang3"
---
---Hiragana scancode.
---
---| "lang4"
---
---Zenkaku/Hankaku scancode.
---
---| "lang5"
---
---The mute key on an American layout.
---
---| "mute"
---
---The volume up key on an American layout.
---
---| "volumeup"
---
---The volume down key on an American layout.
---
---| "volumedown"
---
---The audio next track key on an American layout.
---
---| "audionext"
---
---The audio previous track key on an American layout.
---
---| "audioprev"
---
---The audio stop key on an American layout.
---
---| "audiostop"
---
---The audio play key on an American layout.
---
---| "audioplay"
---
---The audio mute key on an American layout.
---
---| "audiomute"
---
---The media select key on an American layout.
---
---| "mediaselect"
---
---The 'WWW' key on an American layout.
---
---| "www"
---
---The Mail key on an American layout.
---
---| "mail"
---
---The calculator key on an American layout.
---
---| "calculator"
---
---The 'computer' key on an American layout.
---
---| "computer"
---
---The AC Search key on an American layout.
---
---| "acsearch"
---
---The AC Home key on an American layout.
---
---| "achome"
---
---The AC Back key on an American layout.
---
---| "acback"
---
---The AC Forward key on an American layout.
---
---| "acforward"
---
---Th AC Stop key on an American layout.
---
---| "acstop"
---
---The AC Refresh key on an American layout.
---
---| "acrefresh"
---
---The AC Bookmarks key on an American layout.
---
---| "acbookmarks"
---
---The system power scancode.
---
---| "power"
---
---The brightness-down scancode.
---
---| "brightnessdown"
---
---The brightness-up scancode.
---
---| "brightnessup"
---
---The display switch scancode.
---
---| "displayswitch"
---
---The keyboard illumination toggle scancode.
---
---| "kbdillumtoggle"
---
---The keyboard illumination down scancode.
---
---| "kbdillumdown"
---
---The keyboard illumination up scancode.
---
---| "kbdillumup"
---
---The eject scancode.
---
---| "eject"
---
---The system sleep scancode.
---
---| "sleep"
---
---The alt-erase key on an American layout.
---
---| "alterase"
---
---The sysreq key on an American layout.
---
---| "sysreq"
---
---The 'cancel' key on an American layout.
---
---| "cancel"
---
---The 'clear' key on an American layout.
---
---| "clear"
---
---The 'prior' key on an American layout.
---
---| "prior"
---
---The 'return2' key on an American layout.
---
---| "return2"
---
---The 'separator' key on an American layout.
---
---| "separator"
---
---The 'out' key on an American layout.
---
---| "out"
---
---The 'oper' key on an American layout.
---
---| "oper"
---
---The 'clearagain' key on an American layout.
---
---| "clearagain"
---
---The 'crsel' key on an American layout.
---
---| "crsel"
---
---The 'exsel' key on an American layout.
---
---| "exsel"
---
---The keypad 00 key on an American layout.
---
---| "kp00"
---
---The keypad 000 key on an American layout.
---
---| "kp000"
---
---The thousands-separator key on an American layout.
---
---| "thsousandsseparator"
---
---The decimal separator key on an American layout.
---
---| "decimalseparator"
---
---The currency unit key on an American layout.
---
---| "currencyunit"
---
---The currency sub-unit key on an American layout.
---
---| "currencysubunit"
---
---The 'app1' scancode.
---
---| "app1"
---
---The 'app2' scancode.
---
---| "app2"
---
---An unknown key.
---
---| "unknown"
