/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.internal.model;

import static org.hamcrest.CoreMatchers.*;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import org.junit.Test;

import org.springframework.config.java.internal.model.JavaConfigMethod.PrivateMethodError;

import java.lang.reflect.Modifier;

import java.util.ArrayList;


/**
 * Unit tests for {@link ExternalBeanMethod}.
 *
 * @author  Chris Beams
 */
public class ExternalBeanMethodTests {

    // ------------------------------
    // Equivalence tests
    // ------------------------------
    @Test
    public void equivalentMethodsAreEqual() {
        ExternalBeanMethod methodA = new ExternalBeanMethod("foo");
        ExternalBeanMethod methodB = new ExternalBeanMethod("foo");

        assertThat(methodA, equalTo(methodB));
    }

    @Test
    public void methodsWithDifferentModifiersAreNotEqual() {
        ExternalBeanMethod methodA = new ExternalBeanMethod("foo");
        ExternalBeanMethod methodB = new ExternalBeanMethod("foo", Modifier.PUBLIC);

        assertThat(methodA, not(equalTo(methodB)));
    }

    /*
     * creating a new ExternalBeanMethod("foo") is equivalent to a class that declares:
     *
     *     class LineNumberConfig {
     *         @ExternalBean TestBean foo() { ... }
     *     }
     */
    @Test
    public void byDefaultMethodShouldHaveNoModifiers() {
        ExternalBeanMethod method = new ExternalBeanMethod("foo");

        int modifiers = method.getModifiers();

        // 0 signifies 'no modifiers' - see java.lang.reflect.Modifier
        assertEquals(0, modifiers);
    }

    // ------------------------------
    // Validation tests
    // ------------------------------

    @Test
    public void privateExternalBeanMethodsAreNotValid() {
        ArrayList<UsageError> errors = new ArrayList<UsageError>();
        new ExternalBeanMethod("foo", Modifier.PRIVATE).detectUsageErrors(errors);
        assertThat(errors.get(0), instanceOf(PrivateMethodError.class));
    }

}
