/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.grails.data.testing.tck.domains

import grails.persistence.Entity

@Entity
class Publication implements Serializable {

    Long id
    Long version
    String title
    Date datePublished
    Boolean paperback = true

    static mapping = {
        title(index: true)
        paperback(index: true)
        datePublished(index: true)
    }

    static namedQueries = {

        lastPublishedBefore { date ->
            uniqueResult = true
            le('datePublished', date)
            order('datePublished', 'desc')
        }

        recentPublications {
            def now = new Date()
            gt('datePublished', now - 365)
        }

        publicationsWithBookInTitle {
            like('title', 'Book%')
        }

        recentPublicationsByTitle { title ->
            recentPublications()
            eq('title', title)
        }

        latestBooks {
            maxResults(10)
            order('datePublished', 'desc')
        }

        publishedBetween { start, end ->
            between('datePublished', start, end)
        }

        publishedAfter { date ->
            gt('datePublished', date)
        }

        paperbackOrRecent {
            or {
                def now = new Date()
                gt('datePublished', now - 365)
                paperbacks()
            }
        }

        paperbacks {
            eq('paperback', true)
        }

        paperbackAndRecent {
            paperbacks()
            recentPublications()
        }

        thisWeeksPaperbacks() {
            paperbacks()
            def today = new Date()
            publishedBetween(today - 7, today)
        }

        queryThatNestsMultipleLevels {
            // this nested query will call other nested queries
            thisWeeksPaperbacks()
        }
    }
}
