/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jena.geosparql.geo.topological;

import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.FEATURE_A;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.FEATURE_B;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.FEATURE_D;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.GEOMETRY_A;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.GEOMETRY_B;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.GEOMETRY_C_BLANK;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.GEOMETRY_D;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.GEOMETRY_F;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.GEO_FEATURE_Y;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.GEO_FEATURE_Z;
import static org.apache.jena.geosparql.geo.topological.QueryRewriteTestData.TEST_SRS_URI;
import static org.junit.Assert.assertEquals;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.jena.geosparql.configuration.GeoSPARQLConfig;
import org.apache.jena.geosparql.geo.topological.property_functions.simple_features.SfContainsPF;
import org.apache.jena.geosparql.geo.topological.property_functions.simple_features.SfDisjointPF;
import org.apache.jena.geosparql.implementation.index.IndexConfiguration.IndexOption;
import org.apache.jena.geosparql.implementation.index.QueryRewriteIndex;
import org.apache.jena.geosparql.implementation.vocabulary.Geo;
import org.apache.jena.geosparql.spatial.SpatialIndexException;
import org.apache.jena.geosparql.spatial.index.v2.SpatialIndexLib;
import org.apache.jena.graph.Graph;
import org.apache.jena.graph.Node;
import org.apache.jena.graph.NodeFactory;
import org.apache.jena.query.Dataset;
import org.apache.jena.query.QueryExecution;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.query.QuerySolution;
import org.apache.jena.query.ResultSet;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.ModelFactory;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.vocabulary.RDF;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 *
 *
 */
public class GenericPropertyFunctionTest {

    private static Model modelGeoSparql;
    private static Dataset datasetGeoSparql;

    private static Model modelWgs84;
    private static Dataset datasetWgs84;

    public GenericPropertyFunctionTest() {
    }

    @BeforeClass
    public static void setUpClass() throws SpatialIndexException {
        GeoSPARQLConfig.setup(IndexOption.MEMORY, Boolean.TRUE);
        modelGeoSparql = QueryRewriteTestData.addTestDataGeoSparql(ModelFactory.createDefaultModel());
        datasetGeoSparql = SpatialIndexLib.wrapModel(modelGeoSparql, TEST_SRS_URI);

        modelWgs84 = QueryRewriteTestData.addTestDataWgs84(ModelFactory.createDefaultModel());
        datasetWgs84 = SpatialIndexLib.wrapModel(modelWgs84, TEST_SRS_URI);
    }

    @AfterClass
    public static void tearDownClass() {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_geometry() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = GEOMETRY_B.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_geometry_blank() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = GEOMETRY_C_BLANK.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_blank() {

        Graph graph = modelGeoSparql.getGraph();

        Boolean expResult = true;
        String id = GEOMETRY_C_BLANK.asNode().getBlankNodeLabel();
        Node node = NodeFactory.createBlankNode(id);

        Boolean result = graph.contains(node, RDF.type.asNode(), Geo.GEOMETRY_NODE);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_geometry_disabled() {
        GeoSPARQLConfig.setup(IndexOption.MEMORY, Boolean.FALSE);
        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = FEATURE_B.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = false;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);

        GeoSPARQLConfig.setup(IndexOption.MEMORY, Boolean.TRUE);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_feature_geometry() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = FEATURE_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = GEOMETRY_B.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_feature_feature() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = FEATURE_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = FEATURE_B.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_feature() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = FEATURE_B.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_feature_disabled() {

        GeoSPARQLConfig.setup(IndexOption.MEMORY, Boolean.FALSE);
        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = FEATURE_B.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = false;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);

        GeoSPARQLConfig.setup(IndexOption.MEMORY, Boolean.TRUE);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_geometry_false() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = GEOMETRY_D.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = false;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_feature_geometry_false() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = FEATURE_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = GEOMETRY_D.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = false;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_feature_feature_false() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = FEATURE_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = FEATURE_D.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = false;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_feature_false() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = FEATURE_D.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = false;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_geometry_asserted() {


        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = GEOMETRY_F.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_geometry_geometry_asserted_disabled() {

        GeoSPARQLConfig.setup(IndexOption.MEMORY, Boolean.FALSE);
        Graph graph = modelGeoSparql.getGraph();
        Node subject = GEOMETRY_A.asNode();
        Node predicate = Geo.SF_CONTAINS_NODE;
        Node object = GEOMETRY_F.asNode();
        GenericPropertyFunction instance = new SfContainsPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);

        GeoSPARQLConfig.setup(IndexOption.MEMORY, Boolean.TRUE);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_feature_feature_disjoint() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = FEATURE_A.asNode();
        Node predicate = Geo.SF_DISJOINT_NODE;
        Node object = FEATURE_D.asNode();
        GenericPropertyFunction instance = new SfDisjointPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = true;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of queryRewrite method, of class GenericPropertyFunction.
     */
    @Test
    public void testQueryRewrite_feature_feature_disjoint_false() {

        Graph graph = modelGeoSparql.getGraph();
        Node subject = FEATURE_A.asNode();
        Node predicate = Geo.SF_DISJOINT_NODE;
        Node object = FEATURE_B.asNode();
        GenericPropertyFunction instance = new SfDisjointPF();
        QueryRewriteIndex queryRewriteIndex = QueryRewriteIndex.createDefault();
        Boolean expResult = false;
        Boolean result = instance.queryRewrite(graph, subject, predicate, object, queryRewriteIndex);
        assertEquals(expResult, result);
    }

    /**
     * Test of execEvaluated method, of class GenericPropertyFunction.
     */
    @Test
    public void testExecEvaluated_unbound() {


        String query = "PREFIX geo: <http://www.opengis.net/ont/geosparql#>\n"
                + "\n"
                + "SELECT ?subj ?obj\n"
                + "WHERE{\n"
                + "    ?subj geo:sfContains ?obj .\n"
                + "}ORDER by ?subj ?obj";

        List<Resource> subjects = new ArrayList<>();
        List<Resource> objects = new ArrayList<>();
        try (QueryExecution qe = QueryExecutionFactory.create(query, datasetGeoSparql)) {
            ResultSet rs = qe.execSelect();
            while (rs.hasNext()) {
                QuerySolution qs = rs.nextSolution();
                Resource subject = qs.getResource("subj");
                subjects.add(subject);

                Resource object = qs.getResource("obj");
                objects.add(object);
            }
        }

        //Blank nodes limit a value check.
        boolean expResult = true;
        boolean result = subjects.size() == 25 && objects.size() == 25;
        //
        //
        assertEquals(expResult, result);
    }

    /**
     * Test of execEvaluated method, of class GenericPropertyFunction.
     */
    @Test
    public void testExecEvaluated_subject_bound() {


        String query = "PREFIX geo: <http://www.opengis.net/ont/geosparql#>\n"
                + "\n"
                + "SELECT ?obj\n"
                + "WHERE{\n"
                + "    BIND(<http://example.org#FeatureA> AS ?subj) \n"
                + "    ?subj geo:sfContains ?obj .\n"
                + "}ORDER by ?obj";

        List<Resource> objects = new ArrayList<>();
        try (QueryExecution qe = QueryExecutionFactory.create(query, datasetGeoSparql)) {
            ResultSet rs = qe.execSelect();
            while (rs.hasNext()) {
                QuerySolution qs = rs.nextSolution();

                Resource result = qs.getResource("obj");
                objects.add(result);
            }
        }

        //Blank nodes limit a value check.
        int expResult = 6;
        int result = objects.size();
        //
        assertEquals(expResult, result);
    }

    /**
     * Test of execEvaluated method, of class GenericPropertyFunction.
     */
    @Test
    public void testExecEvaluated_subject_bound_geometry() {


        String query = "PREFIX geo: <http://www.opengis.net/ont/geosparql#>\n"
                + "\n"
                + "SELECT ?obj\n"
                + "WHERE{\n"
                + "    BIND(<http://example.org#GeometryA> AS ?subj) \n"
                + "    ?subj geo:sfContains ?obj .\n"
                + "}ORDER by ?obj";

        List<Resource> objects = new ArrayList<>();
        try (QueryExecution qe = QueryExecutionFactory.create(query, datasetGeoSparql)) {
            ResultSet rs = qe.execSelect();
            while (rs.hasNext()) {
                QuerySolution qs = rs.nextSolution();

                Resource result = qs.getResource("obj");
                objects.add(result);
            }
        }

//Blank nodes limit a value check.
        int expResult = 7;
        int result = objects.size();




        assertEquals(expResult, result);
    }

    /**
     * Test of execEvaluated method, of class GenericPropertyFunction.
     */
    @Test
    public void testExecEvaluated_object_bound() {


        String query = "PREFIX geo: <http://www.opengis.net/ont/geosparql#>\n"
                + "\n"
                + "SELECT ?subj\n"
                + "WHERE{\n"
                + "    BIND(<http://example.org#FeatureB> AS ?obj) \n"
                + "    ?subj geo:sfContains ?obj .\n"
                + "}ORDER by ?subj";

        List<Resource> results = new ArrayList<>();
        try (QueryExecution qe = QueryExecutionFactory.create(query, datasetGeoSparql)) {
            ResultSet rs = qe.execSelect();
            while (rs.hasNext()) {
                QuerySolution qs = rs.nextSolution();

                Resource result = qs.getResource("subj");
                results.add(result);
            }
        }

        List<Resource> expResults = Arrays.asList(FEATURE_A, FEATURE_B, GEOMETRY_A, GEOMETRY_B);

        //
        //
        assertEquals(expResults, results);
    }

    /**
     * Test of execEvaluated method, of class GenericPropertyFunction.
     */
    @Test
    public void testExecEvaluated_both_bound() {


        String query = "PREFIX geo: <http://www.opengis.net/ont/geosparql#>\n"
                + "\n"
                + "SELECT ?subj ?obj\n"
                + "WHERE{\n"
                + "    BIND(<http://example.org#FeatureA> AS ?subj) \n"
                + "    BIND(<http://example.org#FeatureB> AS ?obj) \n"
                + "    ?subj geo:sfContains ?obj .\n"
                + "}ORDER by ?subj ?obj";

        List<Resource> subjects = new ArrayList<>();
        List<Resource> objects = new ArrayList<>();
        try (QueryExecution qe = QueryExecutionFactory.create(query, datasetGeoSparql)) {
            ResultSet rs = qe.execSelect();
            while (rs.hasNext()) {
                QuerySolution qs = rs.nextSolution();
                Resource subject = qs.getResource("subj");
                subjects.add(subject);

                Resource object = qs.getResource("obj");
                objects.add(object);
            }
        }

        boolean expResult = true;
        List<Resource> expSubjects = Arrays.asList(FEATURE_A);
        List<Resource> expObjects = Arrays.asList(FEATURE_B);
        boolean result = subjects.equals(expSubjects) && objects.equals(expObjects);
        assertEquals(expResult, result);
    }

    /**
     * Test of execEvaluated method, of class GenericPropertyFunction.
     * @throws SpatialIndexException
     */
    @Test
    public void testExecEvaluated_both_bound_geo() throws SpatialIndexException {
        String query = "PREFIX geo: <http://www.opengis.net/ont/geosparql#>\n"
                + "\n"
                + "SELECT ?subj ?obj\n"
                + "WHERE{\n"
                + "    BIND(<http://example.org#GeoFeatureY> AS ?subj) \n"
                + "    BIND(<http://example.org#GeoFeatureZ> AS ?obj) \n"
                + "    ?subj geo:sfContains ?obj .\n"
                + "}ORDER by ?subj ?obj";

        List<Resource> subjects = new ArrayList<>();
        List<Resource> objects = new ArrayList<>();
        try (QueryExecution qe = QueryExecutionFactory.create(query, datasetWgs84)) {
            ResultSet rs = qe.execSelect();
            while (rs.hasNext()) {
                QuerySolution qs = rs.nextSolution();
                Resource subject = qs.getResource("subj");
                subjects.add(subject);

                Resource object = qs.getResource("obj");
                objects.add(object);
            }
        }

        boolean expResult = true;
        List<Resource> expSubjects = Arrays.asList(GEO_FEATURE_Y);
        List<Resource> expObjects = Arrays.asList(GEO_FEATURE_Z);
        boolean result = subjects.equals(expSubjects) && objects.equals(expObjects);
        assertEquals(expResult, result);
    }

    /**
     * Test of execEvaluated method, of class GenericPropertyFunction.
     * Geo extension: test lookup using geo-literal
     */
    @Test
    public void testExecEvaluated_object_is_literal() {


        String query = "PREFIX geo: <http://www.opengis.net/ont/geosparql#>\n"
                + "\n"
                + "SELECT ?subj\n"
                + "WHERE{\n"
                + "    BIND(\"<http://www.opengis.net/def/crs/EPSG/0/27700> POINT(6 7)\"^^geo:wktLiteral AS ?lit) \n"
                + "    ?subj geo:sfContains ?lit .\n"
                + "}ORDER by ?subj";

        List<Resource> results = new ArrayList<>();
        try (QueryExecution qe = QueryExecutionFactory.create(query, datasetGeoSparql)) {
            ResultSet rs = qe.execSelect();
            while (rs.hasNext()) {
                QuerySolution qs = rs.nextSolution();

                Resource result = qs.getResource("subj");
                results.add(result);
            }
        }

        List<Resource> expResults = Arrays.asList(FEATURE_A, GEOMETRY_A);

        //
        //
        assertEquals(expResults, results);
    }
}
