/*
 * LDAPTemplate.java
 * 
 * Copyright (c) 2001 Sun Microsystems, Inc.
 * All right reserved, all wrongs reversed.
 * 
 * SCCS: @(#) LDAPTemplate.java 1.9 01/09/13 09:24:39
 */
package sunlabs.brazil.contrib;

import java.util.Enumeration;
import java.util.Properties;
import java.util.StringTokenizer;
import sunlabs.brazil.template.Template;
import sunlabs.brazil.template.RewriteContext;
import netscape.ldap.LDAPAttribute;
import netscape.ldap.LDAPAttributeSet;
import netscape.ldap.LDAPModification;
import netscape.ldap.LDAPModificationSet;
import netscape.ldap.LDAPConnection;
import netscape.ldap.LDAPEntry;
import netscape.ldap.LDAPException;

// import netscape.ldap.LDAPSearchConstraints;

import netscape.ldap.LDAPSearchResults;
import netscape.ldap.LDAPv2;

/**
 * The <code>LDAPTemplate</code> is invoked to process LDAP tags embedded in
 * a document.  This version requires the "ldap40.jar" file from the
 * Netscape Navigator distribution.
 * <p>
 * The LDAPTemplate uses the following special tag: <ul>
 * <li> <code>&lt;ldap&gt;</code>
 * </ul>
 * <p>
 * When an LDAP tag is seen, the LDAP database is searched and the results
 * are used to populate the request properties.
 * <p>
 * The following configuration parameters are used to perform the search.
 * The parameters may appear either in the request properties (preceded by
 * the prefix of this template as specified in the configuration file) or as
 * named arguments in the LDAP tag.
 * <dl class=props>
 * <dt> <code>prefix</code>
 * <dd>	The string prefix for the property names that will be stored
 * in the request properties to hold the results.  If not specified,
 * defaults to the prefix of this template as specified in the
 * configuration file.<p>
 * 
 * <dt> <code>dn</code>
 * <dd>	The Distinguished Name (DN) to lookup in the LDAP server.  The format
 * of a DN is described in RFC-1779.  The "dn" and "search" options are
 * mutually exclusive.  When "dn" is specified, only zero or one result
 * will be returned from the LDAP database.  The result (if any) will be
 * stored in the request properties as follows:
 * <pre>
 * &lt;ldap dn="uid=6105,ou=people,o=WebAuth" prefix=<i>name</i>&gt;
 * &lt;property <i>name</i>.dn&gt;
 * &lt;property <i>name</i>.cn&gt;
 * &lt;property <i>name</i>.sn&gt;
 * &lt;property <i>name</i>.objectclass&gt;</pre>
 * etc.  The property <code><i>name</i>.dn</code> is the DN that was
 * found.  Other properties will be defined as shown, based on the
 * attributes present in the LDAP record. <p>
 * 
 * <dt> <code>search</code>
 * <dd> The search filter to use when searching the LDAP server.  The format
 * of a search filter is described in RFC-1558.  The "search" and "dn"
 * options are mutually exclusive.  When "search" is specified, zero or
 * more results will be returned from the LDAP database.  The results
 * will be stored in the request properties as follows:
 * <pre>
 * &lt;ldap search="(givenname=scott)" prefix=<i>name</i>&gt;
 * &lt;property <i>name</i>.rows&gt;
 * 
 * &lt;property <i>name</i>.0.dn&gt;
 * &lt;property <i>name</i>.0.cn&gt;
 * &lt;property <i>name</i>.0.mail&gt;
 * 
 * &lt;property <i>name</i>.1.dn&gt;
 * &lt;property <i>name</i>.1.cn&gt;
 * &lt;property <i>name</i>.1.pager&gt;</pre>
 * etc.  The property <code><i>name</i>.rows</code> is set to the list
 * of record indices found, and can be used by the BSL tag
 * <code>&lt;foreach&nbsp;name=<i>x</i>&nbsp;property=<i>name</i>.rows&gt;</code> to
 * iterate over all records.  Other properties will be defined for
 * each of the records found as shown, based on the attributes present
 * in the each of the LDAP records. <p>
 * 
 * <dt> <code>base</code>
 * <dd> The Distinguished Name of the base record that forms the root of the
 * search tree in the LDAP database.  Used only with the "search" option.
 * Defaults to "".  This would be a good option to specify in the
 * configuration file rather than in the LDAP tag. <p>
 * 
 * <dt> <code>scope</code>
 * <dd> The scope of the LDAP search, one of <ul>
 * <li> "base"	Search only in base record (specified by the "base" option).
 * <li> "one"	Search only records one level below the base record.
 * <li> "sub"	Search the entire subtree below the base record. </ul>
 * Used only with the "search" option.  Defaults to "sub".  This would
 * be a good option to specify in the configuration file rather than in
 * the LDAP tag. <p>
 * 
 * <dt> <code>attributes</code>
 * <dd> The space-delimited list of attribute names to return from the
 * LDAP "dn" or "search" operation.  If empty or unspecified, all
 * attributes for the record are returned.  Not all records in the
 * LDAP database have the same attributes.  Defaults to "". <p>
 * 
 * <dt> <code>host</code>
 * <dd> The hostname of the LDAP server, of the form <code>"host"</code>
 * or <code>"host:port"</code> if the server is not running on the
 * standard LDAP port.  Defaults to "".  This would be a good option to
 * specify in the configuration file rather than in the LDAP tag. <p>
 * 
 * <dt> <code>authenticate</code>
 * <dd> The Distinguished Name used for authenticating to the LDAP server,
 * if necessary.  Defaults to "".  This would be a good option to specify
 * in the configuration file rather than in the LDAP tag. <p>
 * 
 * <dt> <code>password</code>
 * <dd> The password sent when the "authenticate" option is used.  Defaults
 * to "".
 * </dl>
 * 
 * @author	Colin Stevens (colin.stevens@sun.com)
 * @version	1.9, 01/09/13
 */
public class LDAPTemplate extends Template {
    String objectclass_values[] = {
        "top", "person", "organizationalPerson", "inetOrgPerson"
    };

    /**
     * Undocumented Method Declaration.
     * 
     * 
     * @param str
     * 
     * @return
     * 
     * @see
     */
    private static String[] split(String str) {
        if (str == null) {
            return null;
        }

        StringTokenizer st = new StringTokenizer(str);
        int n = st.countTokens();
        String[] strs = new String[n];

        for (int i = 0; i < n; i++) {
            strs[i] = st.nextToken();
        }

        return strs;
    }

    /**
     * Process &lt;ldap&gt; tags.
     */
    public void tag_ldap(RewriteContext hr) {
        String host = hr.get("host", "");
        String auth = hr.get("authenticate", null);
        String password = hr.get("password", null);

        debug(hr);
        hr.killToken();

        String name = hr.get("prefix");

        name = (name == null) ? hr.prefix : name;

        if (name.endsWith(".") == false) {
            name += ".";
        }

        LDAPConnection ld = new LDAPConnection();

        try {
            String dn = hr.get("dn");
			if ( hr.get("verify") != null ) {
            	ld = new LDAPConnection();
            	/* Connect to server */
           		ld.connect( host, LDAPv2.DEFAULT_PORT );
            	/* Authenticate to the server */
            	ld.authenticate( 3, dn, password );
				return;
			} 
            ld.connect(host, LDAPv2.DEFAULT_PORT, auth, password);
			


            if (hr.get("search") != null) {
                String base = hr.get("base", null);
                String str = hr.get("scope", null);
                int scope = LDAPv2.SCOPE_SUB;

                if ("base".equals(str)) {
                    scope = LDAPv2.SCOPE_BASE;
                } else if ("one".equals(str)) {
                    scope = LDAPv2.SCOPE_ONE;
                }

                String[] attrs = split(hr.get("attributes"));

                if (dn != null) {
                    LDAPEntry entry = ld.read(dn, attrs);

                    /*
                     
                     * match was found.
                     */
                    if ((entry != null) && (entry.getDN() != null)) {
                        shove(hr.request.props, name, entry, attrs);
                    }

                    return;
                }

                String search = hr.get("search");

                /*
                 * LDAPSearchConstraints sc = ld.getSearchConstraints();
                 * sc.setMaxResults( 0 );
                 * try {
                 * ld.setOption(LDAPv2.SIZELIMIT,
                 * Integer.decode(hr.get("limit","1000")));
                 * } catch (Exception e) {}
                 */
                StringBuffer sb = new StringBuffer();
                LDAPSearchResults results = ld.search(base, scope, search, 
                                                      attrs, false);

                for (int i = 0; results.hasMoreElements(); i++) {
                    sb.append(i).append(' ');
                    shove(hr.request.props, name + i + ".", results.next(), 
                          attrs);
                }

                hr.request.props.put(name + "rows", sb.toString().trim());
            } else if (hr.get("modattr") != null) {
				// Support delete, modify and add an attribute in one operation
				// .. nodattrs="cn|uid=x" actions="d r a"
				String actions[] = split( hr.get("actions") );
				if ( actions == null ){
					return;
				}
            	LDAPModificationSet attrs = new LDAPModificationSet();
                addattrs(hr.get("modattr"), attrs, actions);
				System.out.println ( attrs );
            	ld.modify( dn, attrs );
            } else if (hr.get("delete") != null) {
                ld.delete(dn);
            } else if (hr.get("add") != null) {
                String add = hr.get("add");

                System.out.println(add);

                LDAPAttributeSet attrs = new LDAPAttributeSet();
                LDAPAttribute attr = new LDAPAttribute("objectclass");

                for (int i = 0; i < objectclass_values.length; i++) {
                    attr.addValue(objectclass_values[i]);
                }

                attrs.add(attr);

                // The add string will look like this: cn=x,y|sn=u,z|x3=p...

                addattrs(add, attrs, null);
        //        attrs.add(new LDAPAttribute("uid", "x1"));

                LDAPEntry myEntry = new LDAPEntry(dn, attrs);

                System.out.println(myEntry);
                ld.add(myEntry);
            }

            ld.disconnect();
        } catch (LDAPException e) {
            hr.request.props.put(name + "error", e.errorCodeToString());

            if (e.getLDAPResultCode() != LDAPException.NO_SUCH_OBJECT) {
                e.printStackTrace();
            }
        }
        finally {
            try {
                ld.disconnect();
            } catch (Exception e) {}
        }
    }

    /**
     * Undocumented Method Declaration.
     * 
     * 
     * @param s
     * @param attr
     * @param attrs
     *
     * @see
     */
    public void addattrs(String s, Object attrs, String actions[]) {
        StringTokenizer st = new StringTokenizer(s, "|");
		LDAPAttribute attr = null;
		int index = 0;

        while (st.hasMoreTokens()) {
            StringTokenizer ts = new StringTokenizer(st.nextToken(), "=,");

            attr = new LDAPAttribute(ts.nextToken());

            while (ts.hasMoreTokens()) {
                attr.addValue(ts.nextToken());
            }

            System.out.println("Add:" + attr);

            if (attrs instanceof LDAPModificationSet) {
				int op; 
				if ( actions[index].equals("a") ) {
					op=LDAPModification.ADD;
				} else if ( actions[index].equals("r") ) {
					op=LDAPModification.REPLACE;
				} else if ( actions[index].equals("d") ) {
					op=LDAPModification.DELETE;
				} else {
					return;
				}
                ((LDAPModificationSet) attrs).add(op, attr);
			} else if (attrs instanceof LDAPAttributeSet) {
                ((LDAPAttributeSet) attrs).add(attr);
            }
			index++;
        }
    }

    /**
     * Undocumented Method Declaration.
     * 
     * 
     * @param props
     * @param prefix
     * @param entry
     * @param names
     * 
     * @see
     */
    private static void shove(Properties props, String prefix, 
                              LDAPEntry entry, String[] names) {
        props.put(prefix + "dn", entry.getDN());

        if (names == null) {
            Enumeration e = entry.getAttributeSet().getAttributes();

            while (e.hasMoreElements()) {
                LDAPAttribute attr = (LDAPAttribute) e.nextElement();

                props.put(prefix + attr.getName(), getAttributeValue(attr));
            }
        } else {
            for (int i = 0; i < names.length; i++) {
                LDAPAttribute attr = entry.getAttribute(names[i]);
                String value = (attr == null) ? "" : getAttributeValue(attr);

                props.put(prefix + names[i], value);
            }
        }
    }

    /**
     * Undocumented Method Declaration.
     * 
     * 
     * @param attr
     * 
     * @return
     * 
     * @see
     */
    private static String getAttributeValue(LDAPAttribute attr) {
        StringBuffer sb = new StringBuffer();
        Enumeration e = attr.getStringValues();

        while (e.hasMoreElements()) {
            sb.append(e.nextElement()).append('|');
        }

        if (sb.length() > 0) {
            sb.setLength(sb.length() - 1);
        }

        return sb.toString();
    }

}

