// Package version contains version information about rq.
package version

import (
	"fmt"
	"runtime"
	"runtime/debug"

	opaversion "github.com/open-policy-agent/opa/v1/version"
)

// These represent the rq version string, in the format
// <major>.<minor>.<patch>(-RC<rc>)?
const major = 0
const minor = 0
const patch = 14
const candidate = 0

// If true, append (dev) to the version string.
const dev = false

// Info represents all of the information relating to the rq version.
type Info struct {
	// Version represents the canonical version string.
	Version string `json:"version"`

	// GoVersion represents the Go version this binary was created with.
	GoVersion string `json:"go_version"`

	// OS represents the operating system this binary is running on.
	OS string `json:"os"`

	// Architecture represents the CPU architecture this binary was built
	// for.
	Architecture string `json:"architecture"`

	// Timestamp is last modified timestamp for the rq git repository at
	// compile time.
	Timestamp string `json:"timestamp"`

	// Revision is the VCS revision string, typically the commit SHA.
	Revision string `json:"revision"`

	// OPA is the version of the embedded OPA library.
	OPA string `json:"opa"`

	// Modified is true if this binary includes un-committed changes.
	Modified bool `json:"modified"`
}

// Version returns the version information for this rq binary.
func Version() Info {
	vs := fmt.Sprintf("%d.%d.%d", major, minor, patch)
	if candidate > 0 {
		vs += fmt.Sprintf("-RC%d", candidate)
	}

	if dev {
		vs += " (dev)"
	}

	vi := Info{
		Version:      vs,
		GoVersion:    runtime.Version(),
		OS:           runtime.GOOS,
		Architecture: runtime.GOARCH,
		OPA:          opaversion.Version,
	}

	if b, ok := debug.ReadBuildInfo(); ok && b != nil {
		for _, s := range b.Settings {
			switch s.Key {
			case "vcs.time":
				vi.Timestamp = s.Value
			case "vcs.revision":
				vi.Revision = s.Value
			case "vcs.modified":
				vi.Modified = s.Value == "true"
			}
		}
	}

	return vi
}
